*! version 1.0.0  29aug2014 dcs

program define dsimih

    version 11.2

    gettoken sub 0 : 0 , parse(" ,")

    local l = length(`"`sub'"')
    
    if "`l'"=="0" {
        disp as error `"subcommand needed"'
        exit 198
    }
    
    if `"`sub'"'==substr("create",1,max(2,`l')) {         /* CReate */
        if inlist("`e(method)'", "Bacchiocchi", "BFanelli", "LLutkepohl") {
            dsimih_create `0'
            exit
        }
        else {
            disp as error `"Cannot determine method used by -svarih-."'
            exit 301
        }
    }
    else if `"`sub'"'==substr("table",1,max(1,`l')) {      // Table
        dsimih_table    `0'
    }
    else if `"`sub'"'==substr("graph",1,max(1,`l')) {      // Graph
        dsimih_graph    `0'
    }
    else if `"`sub'"'==substr("describe",1,max(1,`l')) {   // Describe
        dsimih_describe `0'
    }
    else if `"`sub'"'=="use" {                             // use
        dsimih_use      `0'
    }
    else if `"`sub'"'=="drop" {                            // drop
        dsimih_drop      `0'
    }
    else if `"`sub'"'=="save" {                            // save
        dsimih_save      `0'
    }
    else if `"`sub'"'=="etodta" {                          // etodta
        dsimih_etodta    `0'
    }
    else if `"`sub'"'==substr("modelstats",1,max(1,`l')) { // Modelstats
        dsimih_modelstats `0'
    }
    else {
        di as error "`sub' unknown subcommand"
        exit 198
    }
end

*** ------------------ smaller auxiliary programs -----------------------
// used throughout the subcommands

program define _dsimih_chke, nclass
// checks whether dsimih results are in e(); if not, exits w/ rc==9

    version 11.2

    if "`e(cmd)'"!="svarih" error 301

    capture confirm matrix e(dsimih)
    if _rc {
        disp as txt `"No dynamic simulation results found for -svarih- estimates in e()."'
        exit 9
    }
    if colsof(e(dsimih))==1 { // 'dsimih drop , erase' stores e(dsimih)=J(1,1,.)
        disp as txt `"Previous 'dsimih' results no longer in e()."'
        exit 9
    }

end

program define _dsimih_bsmatexist
// does a c_local for `bsmat'="bsmat" and `bspmat'="bspmat" if they exist
// option short returns locals `bs' and `bsp', with short contents
//   it exists to not collide with programs that have options 'bs' and 'bsp'

    version 11.2

    syntax , [short]

    capture confirm matrix e(dsimih_bs)
    local rc = _rc
    if !`rc' {
        if colsof(e(dsimih_bs))==1 local rc 9
    }
    if !`rc' {
        if "`short'"!="" {
            c_local bs    bs
        }
        else {
            c_local bsmat bsmat
        }
    }

    capture confirm matrix e(dsimih_bsp)
    local rc = _rc
    if !`rc' {
        if colsof(e(dsimih_bsp))==1 local rc 9
    }
    if !`rc' {
        if "`short'"!="" {
            c_local bsp    bsp
        }
        else {
            c_local bspmat bspmat
        }
    }
end

program define _dsimih_parse_saving, rclass

    version 11.2

    capture syntax anything(name=fname) , [replace]
    if _rc > 0 {
        di as err "saving(`0') not valid"
        exit 198
    }   

    mata: ds_pathparts(`"`fname'"')
    if "`r(ext)'"=="" local fname `"`fname'.dta"'

    capture confirm file `"`fname'"'
    if !_rc & "`replace'"=="" error 602
    
    ret local fname "`fname'"
    ret local ireplace `replace'
end

program define _dsimih_tostore, nclass
// converts data set to a format ready to store e(dsimih), and optionally creates a matrix to be stored

    version 11.2

    syntax , depvar(namelist min=2) [matname(name)]

    local neqs : word count `depvar'
    
    tempname irlabel
    label define `irlabel' 1 `: word 1 of `depvar''
    forvalues i=2/`neqs' {
        label define `irlabel' `i' `: word `i' of `depvar'' , add
    }

    encode impulse  , gen(imp_idx) label(`irlabel')
    label values imp_idx
    drop impulse

    encode response , gen(resp_idx) label(`irlabel')
    label values resp_idx
    drop response

    sort regime imp_idx resp_idx step  // same sort order as in _dsimih_main() ; important to sort on numeric values, not on names of `depvar'!
    order regime step imp_idx resp_idx sirf sfevd sirf_seasmp sfevd_seasmp sirf_sebs sfevd_sebs sirf_sebsp sfevd_sebsp

    if `"`matname'"'!="" mkmat _all , matrix(`matname')

end

program define _dsimih_confirmusing, nclass
// checks for existence of file
// determines type: ster or dta
// does two c_locals: `using' contains the file name plus extension ; `usingtype' contains one of "ster", "dta"
//                    note that `using' can have other extensions than .ster or .dta
//                    any other extension than .ster implies `usingtype' "dta"

    version 11.2

    syntax using/
    
    mata: ds_pathparts(`"`using'"')

    if "`r(ext)'"!="" {
        capture confirm file `"`using'"'
        if _rc==601 {
            disp as error `"File '`using'' not found."'
            exit 601
        }
        if `"`r(ext)'"'==".ster" {
            local usingtype ster
        }
        else {
            local usingtype dta
        }
    }
    else {
        capture confirm file `"`using'.dta"'
        if !_rc {
            capture confirm file `"`using'.ster"'
            if !_rc {
                disp as error `"Ambiguous file specification:"'
                disp as error `"Both .ster and .dta files exist for file `using'"'
                exit 198
            }
            local using `"`using'.dta"'
            local usingtype dta
        }
        else {
            capture confirm file `"`using'.ster"'
            if _rc {
                disp as error `"No .ster or .dta file corresponding to `using' found."'
                exit 601
            }
            local using `"`using'.ster"'
            local usingtype ster
        }
    }

    c_local using     `"`using'"'
    c_local usingtype   `usingtype'

end

program define _dsimih_addchars, nclass
// adds dsimih characteristics to data set in memory

    version 11.2

    syntax , stepmat(name) [repmat(name) bs bsp]  // repmat is passed when _addchars is called from _dsimih_create

    if "`bs'"    !="" & "`bsp'"    !="" exit 198
    if "`repmat'"!="" & "`bs'`bsp'"=="" exit 198
    local bstype `bs'`bsp'
    
    foreach curloc in regimes cmdline regimes regimes_Ns method depvar lags mlag exog converged_ml converged_gls idencheck ic_gls Wald_p {
        char _dta[`curloc'] `"`e(`curloc')'"'
    }

    qui mat2mac `stepmat'
    char _dta[stepmat]     `"`r(mat2mac)'"'
    char _dta[stepmat_cn]  `"`: colnames `stepmat''"'
    
    if "`repmat'"!="" {
        tempname repmat_`bstype'
        matrix `repmat_`bstype'' = `repmat'
        
        qui mat2mac `repmat'
        char _dta[repmat_`bstype']    `"`r(mat2mac)'"'
        char _dta[repmat_`bstype'_cn]  "`: colnames `repmat''"
    }
    else {
        _dsimih_bsmatexist // does a c_local `bsmat', `bspmat'    
        if "`bsmat'"!="" {
            qui mat2mac e(dsimih_bs)
            char _dta[repmat_bs]     `"`r(mat2mac)'"'
            char _dta[repmat_bs_cn]   "`: colnames e(dsimih_bs)'"
        }
        if "`bspmat'"!="" {
            qui mat2mac e(dsimih_bsp)
            char _dta[repmat_bsp]    `"`r(mat2mac)'"'
            char _dta[repmat_bsp_cn]  "`: colnames e(dsimih_bsp)'"
        }
    }
    
    qui datasignature set
    char _dta[svarih_version] "`e(version)'"
    char _dta[file_version]   "1.0.0"
    char _dta[created]        "`c(current_date)', `c(current_time)'"
    // cannot add filename of estimation data set as e()-results e.g. can be exported if no data is in memory, just the loaded .ster file

end

program define _dsimih_getstepmat, rclass
// requires a dsimih data set in data set memory
// saves r(stepmat)

    version 11.2

    preserve 

    qui levelsof regime , local(stepmat_regimes)
    mata: st_local("_stepmat_regimes", invtokens(strtoname(tokens(st_local("stepmat_regimes")))))

    local vlist   sirf   sirf_seasmp   sirf_sebs   sirf_sebsp
    local vlist_h sirf_h sirf_seasmp_h sirf_sebs_h sirf_sebsp_h
    foreach curvar of varlist `vlist' {
        qui egen `curvar'_h = max(step) if `curvar'<., by(regime)
    }

    collapse (first) `vlist_h' , by(regime)

    qui keep regime `vlist_h'

    ren sirf_h          stat
    ren sirf_seasmp_h   seasmp
    ren sirf_sebs_h     sebs
    ren sirf_sebsp_h    sebsp

    tempname stepmat
    mkmat _all , matrix(`stepmat')
    matrix rownames `stepmat' = `_stepmat_regimes'

    return matrix stepmat = `stepmat'
    
    restore

end


program define _dsimih_chkdsimfile, nclass
// checks a data set in memory for properties of a dsimih file/data set


    local nvars regime step sirf sirf_seasmp sirf_sebs sirf_sebsp sfevd sfevd_seasmp sfevd_sebs sfevd_sebsp
    foreach curvar of local nvars {
        capture confirm numeric variable `curvar'
        if _rc {
            disp as error `"Not a dsimih data set: numeric variable `curvar' does not exist."'
            exit 111
        }
    }
    local svars impulse response 
    foreach curvar of local svars {
        capture confirm string variable `curvar'
        if _rc {
            disp as error `"Not a dsimih data set: string variable `curvar' does not exist."'
            exit 111
        }
    }
    if c(k)!=12 {
        disp as error `"Not a dsimih data set: may contain only dsimih {it:stats} and their standard errors."'
        exit 111
    }
    
    capture isid regime step impulse response
    if _rc {
        disp as error `"Not a dsimih data set:"'
        disp as error `"  values of variables regime-step-impulse-response do not uniquely identify observations."'
        exit 9
    }
    
end
*** ------------------ end smaller auxiliary programs -------------------






program define dsimih_create

    version 11.2
    
    syntax      [, REGimes(numlist int sort >=0)     ///   default is all regimes
                STep(integer 8)             ///
                noSE                        ///
                SAVing(string)              ///
                                            ///
                bs                          /// 
                bsp                         /// 
                Reps(numlist min=1 max=1 >=3 int)    ///
                BSAving(string)             /// 
                BSBMat(passthru)            ///
                noDots                      ///
                noMAXopts                   ///
                fromb                       ///
                Verbose                     ///
                noTable                     ///
                ]
    
    if "`e(cmd)'" != "svarih" | !inlist("`e(method)'","Bacchiocchi", "BFanelli", "LLutkepohl") {
        error 301
    }
    
    // CHECK INPUT
    if `"`saving'"'!="" {
        _dsimih_parse_saving `saving'
        local savopt  `"saving(`r(fname)', replace)"'
    }
    if `"`bsaving'"'!="" {
        _dsimih_parse_saving `bsaving'
        local bsavopt `"bsaving(`r(fname)', replace)"'
    }

    local mshort = lower(substr("`e(method)'",1,3))
    
    
    // MAKING SURE EXTERNALS ARE EMPTY
    mata: _dsimih_delexternals()
    
    if "`reps'"!="" local repsopt reps(`reps')
    capture noi _dsimih_create , regimes(`regimes') step(`step') `bs' `bsp' `repsopt' `bsavopt' `bsbmat' ///
                                 `se' `dots' `savopt' `maxopts' `fromb' `verbose' `table'
    // note: when commenting out -capture-, last statements of this programm will refer to _rc of last -capture- command

    if "`mshort'"=="bac" {
        local types a b e
    }
    else if "`mshort'"=="bfa" {
        local types b
        forvalues s=2/`e(numregimes)' {
            local types `types' e`s'
        }
    }
    else if "`mshort'"=="llu" {
        local types b l
    }

    local rc = _rc  
    foreach curtype of local types {  // note: this code finishes and constraints still get dropped if user hits -break- key
        capture constraint drop ${T_cns_`curtype'_n}
        macro drop T_cns_`curtype'_n
    }
    mata: _dsimih_delexternals()
    
    if `rc' > 0 {
        exit `rc'
    }

end 

*** --------------------------------- MATA ------------------------------------------------

version 11.2
mata:
mata set matastrict on
    void _dsimih_delexternals() {
        
        real scalar i
        
        string rowvector globvars
        
        globvars = ("T_dsimih","T_dsimih_bs")
        
        for (i=1;i<=cols(globvars);i++) {
            if (findexternal(globvars[i])!=NULL) {
                rmexternal(globvars[i])
            }
        }
        
    }
end




program define dsimih_table, rclass

version 11.2

syntax namelist(name=stat id=stat min=1 max=1) [using/] , ///  need the "/" after "using" for dsimh_use
       [ REGimes(numlist int sort)      ///
         Impulse(string)               ///  contents is varlist, but cannot specify this as the data set with the vars is loaded later; impulse(name) is also not possible, as `string' may contain '?', '-', '*'
         Response(string)              ///
         noCI                          ///
         se                            ///
         SETypes(namelist min=1 max=3) ///
         Level(passthru)               ///
         STep(numlist min=1 >=0 int)   ///
                                       ///
         BYOrder(namelist min=1 max=2) ///
         Format(string)                ///
         *                             /// -list- options
                                       /// 
         Number(passthru)              ///  ster results number ; dsimih_use throws error if `using' is empty or not a .ster file
         Keep                          ///
         clear                         ///
                                       /// 
         /// noFILL                     //  used by -dsimih graph- ; currently not needed, see comment below
       ]

        
        // INPUT CHECKS
        if `"`using'"'=="" {
            _dsimih_chke
        }
        else {
            _dsimih_confirmusing using `using'  // c_locals of `using' (always with file extension) and `usingtype', one of "ster"/"dta"
        }
        
        * _dsimih_chkregimes , regimes(`regimes') c_local // deleted subroutine to check -regimes()- input since -dsimih- subcommands have different default regimes
                                                          // it is easier to do the check in the main routine
        
        capture strtokens `stat' , allowed(SIrf SFevd) comblist(sirf sfevd)
        if _rc==198 {
            disp as error `"{it:stat} incorrectly specified."'
            exit 198
        }
        local stat `r(expanded)'
        
        if "`setypes'"!="" {
            capture strtokens `setypes' , allowed(Asymptotic bs bsp) ///
                    // comblist(asymptotic bs bsp "asymptotic bs" "asymptotic bsp" "bs bsp" "asymptotic bs bsp")  // -strtokens- defaults to permit all combinations
            if _rc==198 {
                disp as error `"Option 'setypes' incorrectly specified."'
                exit 198
            }
            local setypes `r(expanded)'
            local setypes : subinstr local setypes "asymptotic" "asmp"
        }
        else {
            local setypes asmp
        }
        local numse : word count `setypes'
        
        if "`keep'"!="" & `c(changed)' & "`clear'"=="" error 4

        if "`byorder'"!="" {
            capture strtokens `byorder' , allowed(Impulse Response)
            if _rc==198 {
                disp as error `"Option 'byorder' incorrectly specified."'
                exit 198
            }
            local byorder `r(expanded)'
            local numby : word count `byorder'
            forvalues i=1/`numby' {
                local by`i' : word `i' of `byorder'
            }
            local otherby ""
            if "`numby'"=="1" {
                if "`byorder'"=="impulse"  local otherby response
                if "`byorder'"=="response" local otherby impulse
            }
        }

        if `"`format'"'!="" {
            capture confirm numeric format `format'
            if _rc {
                disp as error `"Option 'format' incorrectly specified."'
                exit 198
            }
        }

        qui snapshot save   // using -snapshop- because I need -preserve- later
        local snapshotnum `r(snapshot)'

        capture noisily {  // -capture- b/c of snapshot: must erase temporary file at the end
            
            // IMPULSE AND RESPONSE VARNAMES
            if `"`using'"'!="" {
                qui dsimih_describe using `using' , modelstats nostep
                local depvar `r(depvar)'
            }
            else {
                local depvar `e(depvar)'
            }
            
            if "`impulse'`response'"!="" {
                clear
                foreach curendog of local depvar {
                    qui gen byte `curendog' = .
                }
            }
            if "`impulse'"!="" {
                capture unab imp : `impulse'
                if _rc {
                    disp as error `"Invalid arguments for option 'impulse'."'
                    exit 198
                }
            }
            else {
                local imp `depvar'
            }
            if "`response'"!="" {
                capture unab resp : `response'
                if _rc {
                    disp as error `"Invalid arguments for option 'impulse'."'
                    exit 198
                }
            }
            else {
                local resp `depvar'
            }

            dsimih_use `using', `number' clear `ci' `se' `level'
            
            qui levelsof regime , local(regimes_exist)
            if "`regimes'"=="" {
                local regimes `regimes_exist'
            }
            else {
                local notin : list regimes - regimes_exist
                if "`notin'"!="" {
                    disp as error `"Regimes requested but not in dsimih results: `notin'"'
                    exit 125
                }
            }
            
            qui keep regime impulse response step `stat'*
            
            capture list _all , `options'
            if _rc {
                disp as error `"One or more options not allowed by 'dsimih table' and 'list'."'
                exit 198
            }

            // DELETION OF ENTRIES NOT REQUESTED
            if "`regimes'"!="`e(regimes)'" {
                qui mac2cond regime , comparison(==) macro(`regimes') or
                qui keep if `r(cond)'
            }
            if "`impulse'"!="" {
                qui mac2cond impulse , comparison(==) macro(`imp') or
                qui keep if `r(cond)'
            }
            if "`response'"!="" {
                qui mac2cond response, comparison(==) macro(`resp') or
                qui keep if `r(cond)'
            }
            
            
            if "`step'"!="" {
                qui mac2cond step, comparison(==) macro(`step') or
                qui keep if `r(cond)'
            }

            // DELETION OF VARIABLES NOT REQUESTED
            local sevars ""
            local civars ""
            if "`se'"!="" | "`ci'"!="noci" {
                forvalues i=1/`numse' {
                    local curtype : word `i' of `setypes'
                    if "`se'"!=""     local sevars `sevars' `stat'_se`curtype'
                    if "`ci'"!="noci" {
                        local civars `civars' `stat'_se`curtype'_ci*
                        unab civars : `civars'
                    }
                }
            }

            qui keep regime-response `stat' `sevars' `civars'

            if "`format'"!="" format `stat' `sevars' `civars' `format'

            local secivars `sevars' `civars'
            local secivars_short : subinstr local secivars "`stat'_" "" , all
            foreach curvar of local secivars_short {
                ren `stat'_`curvar' `curvar'
            }
            
            if "`byorder'"=="" {
                sort regime impulse response step
                if `"`options'"'=="" {
                    list regime impulse response step `stat' `secivars_short' , noobs sepby(impulse response)
                }
                else {
                    list regime impulse response step `stat' `secivars_short' , `options'
                }
            }
            else {
                foreach rgm of local regimes {
                    preserve
                    qui keep if regime==`rgm'
                    sort `byorder' `otherby' step
                    if `numby'==2 {
                        if `"`options'"'=="" {
                            by `byorder' : list regime `otherby' step `stat' `secivars_short' , noobs sep(0)
                        }
                        else {
                            by `byorder' : list regime `otherby' step `stat' `secivars_short' , `options'
                        }
                    }
                    else {
                        if `"`options'"'=="" {
                            by `byorder' : list regime `otherby' step `stat' `secivars_short' , noobs sepby(`otherby')
                        }
                        else {
                            by `byorder' : list regime `otherby' step `stat' `secivars_short' , `options'
                        }
                    }
                    restore
                }
            }
            
            foreach curvar of local secivars_short {
                ren `curvar' `stat'_`curvar'
            }
        }
        
        if _rc {
            snapshot restore `snapshotnum'
            snapshot erase `snapshotnum'
            exit _rc
        }
       
        if "`keep'"=="" snapshot restore `snapshotnum'
        snapshot erase `snapshotnum'
       
end




program define dsimih_graph, nclass

version 11.2

syntax namelist(name=stat id=stat min=1 max=1) [using] , ///  not 'using/' since it is passed to dsimih_table
       [ REGimes(numlist int sort)            ///
         Impulse(passthru)                   ///
         Response(passthru)                  ///
         noCI                                ///
         SEType(namelist min=1 max=1)        ///
         Level(numlist min=1 max=1 >0 <100)  ///
         LSTep(numlist min=1 max=1 >=0 int)  ///
         USTep(numlist min=1 max=1 >=0 int)  ///
                                             ///
         BYOrder(namelist min=1 max=2)       ///
         PLotopts(string)                    ///
         CIOPts(string)                      ///
         BYOPts(string)                      ///
         *                                   /// twoway options
                                             ///
         Number(passthru)                    ///  ster results number ; dsimih_use throws error if `using' is empty or not a .ster file
         keep                                ///
         clear                               ///
       ]

        // INPUT CHECKS
        // e(dsimih) or alternatively `using' are checked in dsimih_table
        // regimes are checked in dsimih_table call
        
        capture strtokens `stat' , allowed(SIrf SFevd) comblist(sirf sfevd)
        if _rc==198 {
            disp as error `"{it:stat} incorrectly specified."'
            exit 198
        }
        local stat `r(expanded)'

        if "`setype'"!="" {
            capture strtokens `setype' , allowed(Asymptotic bs bsp) comblist(asymptotic bs bsp)
            if _rc==198 {
                disp as error `"Option 'setype' incorrectly specified."'
                exit 198
            }
            local setype `r(expanded)'
            local setype_sh `setype'
            if "`setype'"=="asymptotic" local setype_sh "asmp"
        }
        else {
            local setype_sh asmp
        }

        if "`keep'"!="" & `c(changed)' & "`clear'"=="" error 4

        if "`regimes'"!="" local regimeopt reg(`regimes')

        if "`level'"!="" {
            local lopt     l(`level')
        }
        else {
            local level `c(level)'
        }
            
        if "`lstep'"=="" local lstep 0
        if "`ustep'"=="" {
            local stepopt ""
        }
        else {
            if `ustep'<`lstep' {
                disp as error `"Argument for option 'ustep' must be >= 'lstep'."'
                exit 198
            }
            local stepopt step(`lstep'/`ustep')
        }
        if "`byorder'"!="" {
            capture strtokens `byorder' , allowed(Impulse Response)
            if _rc==198 {
                disp as error `"Option 'byorder' incorrectly specified."'
                exit 198
            }
            local byorder `r(expanded)'
        }
        else {
            local byorder impulse response
            local byoopt  byo(`byorder')
        }
        if "`setype'"!=""   local seopt    setype(`setype')
        
        _dsimih_graph_byparse , `byopts'
        if (`"`r(iscale)'"' == "") local iscale iscale(*.75)

        preserve
        
        qui dsimih_table `stat' `using', `regimeopt' `impulse' `response' `lopt' `seopt' `stepopt' `byoopt' `number' keep clear `ci'

        if "`ci'"!="noci" {
            graph twoway (rarea `stat'_se`setype_sh'_cil`level' `stat'_se`setype_sh'_ciu`level' step, ///
                                 pstyle(ci) `ciopts' yvarlabel(`"`level'% CI"' `"`level'% CI"') ) || ///
                         (line  `stat' step, `plotopts' pstyle(p1line) ) ///
                         , by(regime `byorder', `byopts' `iscale')        ///
                           ytitle("") ylabels(, angle(horizontal)) `options' `saving' `name'
        }
        else {
            graph twoway (line  `stat' step, `plotopts' pstyle(p1line) ) ///
                         , by(regime `byorder', `byopts' `iscale') ytitle("") `options' `saving' `name'
        }


        if "`keep'"!="" restore, not

end

*** --------------------------------- SUBROUTINES -----------------------------------------

program _dsimih_graph_byparse, rclass
    syntax , [ iscale(string asis) * ]

    return local iscale `"`iscale'"'
    return local byopts `"`options'"'
end




program define dsimih_describe, rclass

    version 11.2

    syntax [ using/ ] , /// need the "/" after "using" for dsimh_use
           [ Modelstats ///
             Cmdline    ///
             noSTep     ///
             Bootstrap  ///
             All        ///
             Number(passthru) /// ster results number ; throws error if `using' is empty or not a .ster file
           ]

    if "`all'"!="" {
        local modelstats modelstats
        local cmdline    cmdline
        local step       step
        local bootstrap  bootstrap
    }
    
    if "`step'"!="nostep" {  // redefine local step to make conditions more intuitive
        local step step
    }
    else {
        local step ""
    }

    if `"`using'"'!="" {
        _dsimih_confirmusing using `using' // c_locals of `using' (always with file extension) and `usingtype', one of "ster"/"dta"
        if "`usingtype'"=="ster" {
            tempname ehold
            _estimates hold `ehold' , nullok restore
            estimates use `using' , `number'
        }
    }

    if `"`number'"'!="" & (`"`using'"'=="" | "`usingtype'"!="ster") {
        if `"`number'"'!="" {
            disp as error `"Option 'number' only allowed when loading referring to a .ster file."'
            exit 198
        }
    }

    if `"`using'"'=="" | "`usingtype'"=="ster" {
        if "`e(cmd)'"!="svarih" error 301
        
        if "`modelstats'"!="" { // if a macro result does not exist, nothing happens, and the new local is empty
            foreach curloc in regimes regimes_Ns method depvar lags mlag exog converged_ml converged_gls idencheck ic_gls Wald_p {
                local `curloc' `e(`curloc')'
            }
        }

        if "`cmdline'"!="" {
            qui est describe
            local cmdline `"`r(cmdline)'"'
        }

        if "`step'`bootstrap'"!="" {
            capture _dsimih_chke
            if _rc {
                local nodisimih nodisimih
                local step      ""
                local bootstrap ""
            }
        }

        if "`step'"!="" {
        
            local matnames stepmat
            
            preserve        // have a double preserve here: before and within _getstepmat
                            // necessary b/c _getstepmat requires that the dsimih data set be in memory, b/c of call to _getstepmat from _dsimih_create
            dsimih_use , clear noci se  
            tempname stepmat
            _dsimih_getstepmat
            matrix `stepmat' = r(stepmat)
            restore

        }

        if "`bootstrap'"!="" {
        
            _dsimih_bsmatexist , short // returns locals `bs'="bs" and `bsp'="bsp" if they exist

            if "`bs'`bsp'"!="" {

                if "`bs'" !="" local matnames `matnames' repmat_bs
                if "`bsp'"!="" local matnames `matnames' repmat_bsp

                tempname repmat_bs repmat_bsp
                foreach bstype in `bs' `bsp' {
                    matrix `repmat_`bstype'' = e(dsimih_`bstype')
                }
            }
        }

    }
    else {

        preserve

        dsimih_use `using' , clear noci se
        
        disp ""
        disp as text "dsimih file, created `: char _dta[created]'"
        
        if "`modelstats'"!="" {
            foreach curloc in regimes regimes_Ns method depvar lags mlag exog converged_ml converged_gls idencheck ic_gls Wald_p {
                local `curloc' `: char _dta[`curloc']'
            }
        }

        if "`cmdline'"!="" {
            local cmdline `: char _dta[cmdline]'
        }

        if "`step'"!="" local matnames stepmat
        if "`bootstrap'"!="" {
            local jnk : char _dta[repmat_bs_cn]
            if "`jnk'"!="" local matnames `matnames' repmat_bs
            local jnk : char _dta[repmat_bsp_cn]
            if "`jnk'"!="" local matnames `matnames' repmat_bsp
        }
        
        if "`matnames'"!="" {
            foreach curmat of local matnames {
                tempname `curmat'
                matrix ``curmat'' = `: char _dta[`curmat']'
                matrix colnames ``curmat'' = `: char _dta[`curmat'_cn]'
            }
            
            qui mat2mac ``: word 1 of `matnames''' , col(1)
            local stepmat_regimes `r(mat2mac)'
            mata: st_local("_stepmat_regimes", invtokens(strtoname(tokens(st_local("stepmat_regimes")))))
            
            foreach curmat of local matnames {
                matrix rownames ``curmat'' = `_stepmat_regimes'
            }

        }
        restore
        
    }

    // RESULTS DISPLAY

    disp ""
    
    if "`modelstats'"!="" {
    
        disp as text "{cmd:svarih} model:" _n
        foreach curloc in depvar regimes lags {
            local `curloc'str   ``curloc''
            if length("``curloc''")>50 local `curloc'str = substr("``curloc''", 1, 50) + ".."
        }

        local numdepvar  : word count `depvar'
        local numregimes : word count `regimes'
        local numlags    : word count `lags'
        
        disp as text   "{col 7}svarih method "  as res "`e(method)'"
        
        if "`idencheck'"!="" disp as text "{col 7}check for local identification : " as result "`idencheck'"
        if "`Wald_p'"   !="" disp as text "{col 7}check for global identification, Prob > chi2(1) : " as result %9.3f `Wald_p'
        
        if `converged_ml'==1 {
            disp as text "{col 7}ML convergence declared"
        }
        else {
            disp as text "{col 7}ML convergence {bf:NOT} declared"
        }
        
        if "`e(glsiter)'"!="" {
            if "`e(glsiter)'"=="0" {
                disp as text "{col 7}no GLS iterations"
            }
            else {
                if e(converged_gls)==1 {
                    disp as text "{col 7}GLS convergence declared"          _c
                }
                else {
                    disp as text "{col 7}GLS convergence {bf:NOT} declared" _c
                }
                disp as text " after `e(ic_gls)' GLS iterations"
            }
        }

        disp ""
        disp as res    "  " %3.0f `numdepvar'  as text       " dep.var.   : " as result "`depvarstr'"
        disp as res    "  " %3.0f `numregimes' as text       " regimes    : " as result "`regimesstr'"
        disp                                   as text "{col 7}  obs/rgm  : " as result "`regimes_Ns'"
        disp as res    "  " %3.0f `numlags'    as text       " lags       : " as result "`lagsstr'"

        disp                                   as text "{col 7}  max.lag  : " as result "`e(mlag)'"

        if "`e(exog)'"!="" {
            local numexog : word count `exog'
            local exogstr   `exog'
            if length("`exog'")>50 local exogstr = substr("`exog'", 1, 50) + ".."
            disp as res "  " %3.0f `numexog'    as text " exog.var.  : " as result "`exogstr'"
        }
        
        disp ""
    }

    if "`cmdline'"!="" {
        disp as text  "{cmd:svarih} estimates generated by:" _n
        
        local cmdpart : piece 1 60 of `"`cmdline'"'
        disp as text `"{col 3}`cmdpart'"'
        local i 2
        while `"`cmdpart'"'!="" {
            local cmdpart : piece `i' 60 of `"`cmdline'"'
            disp as text `"{col 13}`cmdpart'"'
            local ++i
        }
    }
    
    if "`nodisimih'"!="" {
        disp as text "No {cmd:dsimih} results present in {cmd:e()}."
    }
    
    if "`matnames'"!="" {
        if "`stepmat'"   !="" matlist `stepmat'    , names(column) title("Dynamic simulation step size") noblank
        capture confirm matrix `repmat_bs'
        if !_rc matlist `repmat_bs'  , names(column) title("# replications for {it:setype}='bs'")
        capture confirm matrix `repmat_bsp'
        if !_rc matlist `repmat_bsp' , names(column) title("# replications for {it:setype}='bsp'")
    }


    // RETURN RESULTS
    if "`modelstats'"!="" {
        foreach curloc in method depvar regimes regimes_Ns lags exog idencheck {
            return local `curloc' `"``curloc''"'
        }

        foreach curscl in mlag converged_ml converged_gls ic_gls Wald_p {
            if "``curscl''"!="" return scalar `curscl' = ``curscl''
        }
    }

    if "`cmdline'"!="" {
        return local cmdline `"`cmdline'"'
    }
    
    if "`matnames'"!="" {
        if "`step'"!="" qui mat2mac `stepmat' , col(1)
        return local stepmat_regimes "`r(mat2mac)'"

        foreach curmat of local matnames {
            return matrix `curmat' = ``curmat''
        }
    }

end




program define dsimih_use, nclass

version 11.2

syntax   [ anything(name=usingfile) ] ,            /// just the filename, without "using" ; name=using is not allowed ; see [U12], section 11.3
         [ noCI                                    /// 
           se                                      ///
           SETypes(namelist min=1 max=3)           ///
           Level(numlist min=1 max=1 >=10 <=99.99) ///
           clear                                   ///
           Number(passthru)                        /// ster results number ; throws error if `using' is empty or not a .ster file
                                    /// undocumented internal options
           mata(string)             /// one of "nose", "asmp", "bs", "bsp" ; pulls dsimih matrix from Mata
           bsreps                   /// load T_dsimih_bs data into data set memory
         ]

    local using `"`usingfile'"'
    
    if "`e(cmd)'"!="svarih" & `"`using'"'=="" & "`mata'"=="" error 301

    if "`clear'"=="" & c(changed)==1 error 4
    
    if `"`using'"'!="" {
        _dsimih_confirmusing using `using' // c_locals of `using' (always with file extension) and `usingtype', one of "ster"/"dta"
        if "`usingtype'"=="ster" {
            tempname ehold
            _estimates hold `ehold' , nullok restore
            estimates use `using' , `number'
        }
    }

    if `"`number'"'!="" & (`"`using'"'=="" | "`usingtype'"!="ster") {
        if `"`number'"'!="" {
            disp as error `"Option 'number' only allowed when referring to a .ster file."'
            exit 198
        }
    }
    
    local deltypes ""
    if "`setypes'"!="" {
        capture strtokens `setypes' , allowed(Asymptotic bs bsp) ///
                // comblist(asymptotic bs bsp "asymptotic bs" "asymptotic bsp" "bs bsp" "asymptotic bs bsp")  // -strtokens- defaults to permit all combinations
        if _rc==198 {
            disp as error `"Option 'setypes' incorrectly specified."'
            exit 198
        }
        local setypes `r(expanded)'
        local setypes : subinstr local setypes "asymptotic" "asmp"
        
        local alltypes asmp bs bsp
        local deltypes : list alltypes - setypes
    }
    
    preserve
    clear
    
    if `"`using'"'=="" | "`usingtype'"=="ster" {
        if `"`mata'"'=="" {

            _dsimih_chke

            tempname dsmat
            matrix `dsmat' = e(dsimih)

            qui svmat double `dsmat', names(col)
            qui recast long regime step , force
        }
        else {
            mata: _dsimih_use_T("`bsreps'")  // `bsreps' is either "bsreps" or "", not a number
            
            if "`bsreps'"=="" {
                local setypes asmp bs bsp
                if "`mata'"=="nose" {
                    local othertypes asmp bs bsp
                }
                else {
                    local othertypes : list setypes - mata
                    ren sirf_se   sirf_se`mata'
                    ren sfevd_se  sfevd_se`mata'
                }
                foreach stat in sirf sfevd {
                    foreach curtype of local othertypes {
                        qui gen double `stat'_se`curtype' = .
                    }
                }
            }
        }
        
        local depvar `e(depvar)'
        local neqs : word count `depvar'
        tempname irlabel
        label define `irlabel' 1 `: word 1 of `depvar''
        forvalues i=2/`neqs' {
            label define `irlabel' `i' `: word `i' of `depvar'' , add
        }

        label values imp_idx  `irlabel'
        decode imp_idx, gen(impulse)
        drop imp_idx

        label values resp_idx  `irlabel'
        decode resp_idx, gen(response)
        drop resp_idx

        label variable regime         "regime #"    
        label variable step          "forecast step/horizon"
        label variable impulse       "impulse variable"
        label variable response      "response variable"
        label variable sirf          "SIRF"
        label variable sfevd         "SFEVD"

        capture label variable sirf_seasmp   "SIRF: asymptotic s.e."
        capture label variable sirf_sebs     "SIRF: s.e., parametric bootstrap"
        capture label variable sirf_sebsp    "SIRF: s.e., residual bootstrap"
        capture label variable sfevd_seasmp  "SIRF: asymptotic s.e."
        capture label variable sfevd_sebs    "SIRF: s.e., parametric bootstrap"
        capture label variable sfevd_sebsp   "SIRF: s.e., residual bootstrap"
    }
    else {
    
        qui use `usingfile'
        
        _dsimih_chkdsimfile
        capture datasignature confirm
        if _rc==459 {
            disp as error `"dsimih files must have a {help datasignature} set."'
            exit 459
        }
        else if _rc>0 {
            disp as error `"Data in dsimih file have been modified."'
            exit 9
        }

        /*
        fillin regime impulse response step
        qui drop _fillin
        sort regime impulse response step
        */

    }

    if "`ci'"!="noci" {  // option 'noci' has not been used
        if "`level'"=="" local level `c(level)'
        
        tempname half cv
        scalar `half' = (1-(`level' / 100)) / 2
        scalar `cv' = invnormal(1-`half')
        local l = round(`level')

        foreach stat in sirf sfevd {
            foreach setype in asmp bs bsp {
                qui gen `stat'_se`setype'_cil`l' = `stat' - `stat'_se`setype' * `cv'
                qui gen `stat'_se`setype'_ciu`l' = `stat' + `stat'_se`setype' * `cv'
            }
        }
        order regime step impulse response ///
              sirf  sirf_seasmp  sirf_seasmp_cil`l'  sirf_seasmp_ciu`l'  ///
                    sirf_sebs    sirf_sebs_cil`l'    sirf_sebs_ciu`l'    ///
                    sirf_sebsp   sirf_sebsp_cil`l'   sirf_sebsp_ciu`l'   ///
              sfevd sfevd_seasmp sfevd_seasmp_cil`l' sfevd_seasmp_ciu`l' ///
                    sfevd_sebs   sfevd_sebs_cil`l'   sfevd_sebs_ciu`l'   ///
                    sfevd_sebsp  sfevd_sebsp_cil`l'  sfevd_sebsp_ciu`l'

        label variable sirf_seasmp_cil`l'  "SIRF: `l'% lower asymptotic band"
        label variable sirf_seasmp_ciu`l'  "SIRF: `l'% upper asymptotic band"
        label variable sirf_sebs_cil`l'    "SIRF: `l'% lower band, parametric bootstrap"
        label variable sirf_sebs_ciu`l'    "SIRF: `l'% upper band, parametric bootstrap"
        label variable sirf_sebsp_cil`l'   "SIRF: `l'% lower band, residual bootstrap"
        label variable sirf_sebsp_ciu`l'   "SIRF: `l'% upper band, residual bootstrap"
        label variable sfevd_seasmp_cil`l' "SFEVD: `l'% lower asymptotic band"
        label variable sfevd_seasmp_ciu`l' "SFEVD: `l'% upper asymptotic band"
        label variable sfevd_sebs_cil`l'   "SFEVD: `l'% lower band, parametric bootstrap"
        label variable sfevd_sebs_ciu`l'   "SFEVD: `l'% upper band, parametric bootstrap"
        label variable sfevd_sebsp_cil`l'  "SFEVD: `l'% lower band, residual bootstrap"
        label variable sfevd_sebsp_ciu`l'  "SFEVD: `l'% upper band, residual bootstrap"
        
        if "`deltypes'"!="" {
            foreach curtype of local deltypes {
                qui drop *_se`curtype'_ci*
            }
        }
        
    }
    else {
        order regime step impulse response sirf* sfevd*
    }
    
    if "`se'"=="" {
        qui drop *_seasmp *_sebs *_sebsp
    }
    else {
        if "`deltypes'"!="" {
            foreach curtype of local deltypes {
                qui drop *_se`curtype'
            }
        }
    }
    
    restore, not
    mata: st_updata(0)  // c(changed) is 1 from the actions of dsimih_use: the data set in memory is not what is stored in the .dta/.ster

end


*** --------------------------------- MATA ------------------------------------------------

version 11.2
mata:
mata set matastrict on
    void _dsimih_use_T(| string scalar bsreps) {
        
        string rowvector vtypes, vnames
        
        real scalar reps, reprows, i, okidx
        
        real colvector repsok
        
        real matrix dsimih, jnk
        
        pointer(real matrix) colvector pDsimih

        if (bsreps=="") {
            dsimih = **findexternal("T_dsimih")
            vnames = ("regime", "step", "imp_idx", "resp_idx", "sirf"  , "sfevd" , "sirf_se", "sfevd_se")
            vtypes = ("long" , "long", "byte"   , "byte"    , "double", "double", "double" , "double")
        }
        else {

            pDsimih = *findexternal("T_dsimih_bs")
            reps = rows(pDsimih)
            repsok = J(reps, 1, .)
            for (i=1;i<=reps;i++) {
                if ( (pDsimih[i]!=NULL) ) {
                    repsok[i] = i
                }
            }
            repsok = select(repsok, repsok:<.)
            reprows = rows(*(pDsimih[repsok[1]]))
            dsimih = J(rows(repsok)*reprows, 8, .)
            
            for (i=1;i<=rows(repsok);i++) {
                dsimih[|(i-1)*reprows+1,1 \ i*reprows,8|] = *(pDsimih[repsok[i]])
            }
            dsimih = (dsimih , (repsok # J(reprows, 1, 1)))
            vnames = ("regime", "step", "imp_idx", "resp_idx", "sirf"  , "sfevd" , "sirf_se", "sfevd_se", "reps")
            vtypes = ("long" , "long", "byte"   , "byte"    , "double", "double", "double" , "double"  , "long")
        }
        
        jnk = st_addvar(vtypes, vnames)
        st_addobs(rows(dsimih))
        st_store(., (1..cols(vnames)), dsimih)
    }
end





program define dsimih_drop, eclass

version 11.2

syntax , [ REGimes(numlist min=1 int sort)  /// note: default is no regimes, not all regimes, like in the other subcommands
           Maxstep(numlist min=1 max=1 >=0) ///
           SEType(name)                     /// asmp/bs/bsp
           erase ]                          //  erases all dsimih-related results from e()

    _dsimih_chke
    _dsimih_bsmatexist , short // returns locals `bs'="bs" and `bsp'="bsp" if they exist

    qui dsimih_describe , step
    local regimes_exist `r(stepmat_regimes)'
    if "`regimes'"!="" {   // note: dsimih_drop does not have all regimes as the default
        local notin : list regimes - regimes_exist
        if "`notin'"!="" {
            disp as error `"Regimes specified but not in dsimih results: `notin'"'
            exit 125
        }
    }

    tempname erasemat
    matrix `erasemat' = J(1,1,.)
    if "`erase'"!="" {
        if "`regimes'`maxstep'`setype'"!="" {
            disp as error `"Option 'erase' cannot be used in conjunction with other options."'
            exit 198
        }
        matrix `erasemat' = J(1,1,.)
        ereturn matrix dsimih = `erasemat' , copy
        if "`bs'"!=""  ereturn matrix dsimih_bs = `erasemat' , copy
        if "`bsp'"!="" ereturn matrix dsimih_bsp = `erasemat', copy
        exit
    }
    else {
        if `"`regimes'`maxstep'`setype'"'=="" {
            disp as error `"At least one option required."'
            exit 198
        }
    }

    if "`setype'"!="" {
        capture strtokens `setype' , allowed(Asymptotic bs bsp) comblist(asymptotic bs bsp)
        if _rc==198 {
            disp as error `"Option 'setype' incorrectly specified."'
            exit 198
        }
        local setype `r(expanded)'
        if "`setype'"=="asymptotic" local setype "asmp"
    }

    preserve

    dsimih use, clear noci se
    
    if "`maxstep'"!="" {
        local stepcond "step > `maxstep'"
        local amp " & "
        local if  " if "
    }
    if "`setype'"!="" {
        if "`regimes'"!="" {
            foreach s of local regimes {
                qui replace sirf_se`setype'  = . if regime==`s' `amp' `stepcond'
                qui replace sfevd_se`setype' = . if regime==`s' `amp' `stepcond'
            }
        }
        else {
            qui replace sirf_se`setype'  = . `if' `stepcond'
            qui replace sfevd_se`setype' = . `if' `stepcond'
        }
    }
    else {
        if "`regimes'"!="" {
            foreach s of local regimes {
                qui drop if regime==`s' `amp' `stepcond'
            }
        }
        else {
            qui drop if `stepcond'   // cannot be missing since 1 option has to be specified
        }
    }

    qui count 
    if `r(N)'==0 {
        ereturn matrix dsimih             = `erasemat' , copy
        if "`bs'" !="" ereturn matrix dsimih_bs  = `erasemat' , copy
        if "`bsp'"!="" ereturn matrix dsimih_bsp = `erasemat' , copy
        exit
    }
    
    local depvar `e(depvar)'
    tempname newmat
    _dsimih_tostore, depvar(`depvar') matname(`newmat')
    
    // ADJUST e(dsimih_bs) , e(dsimih_bsp)
    // keep rows of e(dsimih_`bstype') if a corresponding row exists in `stepmat' and if `bstype' has a nonmissing value
    tempname stepmat jnk
    _dsimih_getstepmat
    matrix `stepmat' = r(stepmat)

    qui mat2mac `stepmat' , col(1)
    local regimes_keep `r(mat2mac)'

    tempname bsmat_old bsmat_new
    if "`bs'`bsp'"!="" {
        foreach bstype in `bs' `bsp' {
            qui count if sirf_se`bstype' < .
            if r(N)==0 {
                ereturn matrix dsimih_`bstype' = `erasemat' , copy
            }
            else {
                matrix `jnk' = `stepmat'[1..., "se`bstype'"]
                qui mat2mac `jnk' , col(1)
                local hasobs `r(mat2mac)'
                matrix `bsmat_old' = e(dsimih_`bstype')
                local cnames ""
                forvalues i=1/`: word count `regimes_keep'' {
                    local rgm : word `i' of `regimes_keep'
                    local dot : word `i' of `hasobs'
                    if `dot'<. {
                        matrix `bsmat_new' = ( nullmat(`bsmat_new') \ `bsmat_old'[rownumb(`bsmat_old', "_`rgm'"), 1...] )
                        local cnames `cnames' _`rgm'
                    }
                }
                matrix colnames `bsmat_new' = `: colnames `bsmat_old''
                matrix rownames `bsmat_new' = `cnames'

                ereturn matrix dsimih_`bstype' = `bsmat_new'
            }
        }
    }

    restore

    ereturn matrix dsimih = `newmat'

end




program define dsimih_save, nclass

version 11.2

syntax   anything        ///
       [ , replace]

    preserve
    
    capture drop sirf_se*_ci*
    capture drop sfevd_se*_ci*

    * _dsimih_chkdsimfile
    
    save `"`anything'"' , `replace'

    restore
    
end




program define dsimih_etodta, nclass

version 11.2

syntax   anything        ///
       [ , replace]

    _dsimih_chke
    
    preserve
    
    qui dsimih_use , se noci clear
    
    tempname stepmat repmat_bs repmat_bsp
    
    _dsimih_getstepmat
    matrix `stepmat' = r(stepmat)

    _dsimih_addchars , stepmat(`stepmat')
    
    label data "dsimih results file"
    dsimih save `anything' , `replace'

    restore
    
end




program define mat2mac, rclass

    version 10.1
    syntax anything(name=matin) ,              ///
           [ Row(numlist min=1 max=1 >0 int)     ///
             Column(numlist min=1 max=1 >0 int)  ///
             comma                               ///
           ]

    confirm matrix `matin'

    if `: word count `matin''!=1 {
        disp as error `"You may only supply one {it:matname}."'
        exit 198
    }

    tempname matname
    matrix `matname' = `matin'  // copy matrix to make sure ado works for r() and e() matrices

    local col `column'

    foreach curdim in row col {
        if "``curdim''"!="" {
            local dimsize = `curdim'sof(`matname')
            if `dimsize' < ``curdim'' {
                disp as error `"Argument of option '`curdim'' out of bounds."'
                exit 125
            }
        }
    }
    
    if "`row'`col'"=="" {
        local numrows = rowsof(`matname')
        forvalues i=1/`numrows' {
            
            mata: _mat2mac_extract("`matname'", `i', 0    )  // returns local `matastring'
            local matastring : subinstr local matastring " " ", " , all

            if `numrows'==1 {
                local matstr "( `matastring' )"
            }
            else if `i'==1 {
                local matstr "( `matastring' \ "
            }
            else if `i'<`numrows' {
                local matstr  "`matstr'`matastring' \ "
            }
            else {
                local matstr  "`matstr'`matastring' )"
            }
        }
    }
    else if "`row'"=="" {
        mata: _mat2mac_extract("`matname'",     0, `col')
        local matstr `matastring'
        if "`comma'"!="" local matstr : subinstr local matstr " " ", " , all
    }
    else if "`col'"=="" {
        mata: _mat2mac_extract("`matname'", `row', 0    )
        local matstr `matastring'
        if "`comma'"!="" local matstr : subinstr local matstr " " ", " , all
    }
    else {
        mata: _mat2mac_extract("`matname'", `row', `col')
        local matstr `matastring'
    }
    
    return local mat2mac `matstr'
    
    if length(`"`matstr'"')>60 local matstr = substr(`"`matstr'"',1,56) + " ..."
    disp as text _n "Extracted: " as result "`matstr'"

end

*** --------------------------------- MATA ------------------------------------------------

version 10.1
mata
mata set matastrict on
void _mat2mac_extract(string scalar matname, real scalar row, real scalar col) {
    // extracts row or column or scalar of a matrix and stores it in macro `matastring'
    
    real scalar size,
                i

    string scalar result

    transmorphic matrix source,
                        submat
    
    source = st_matrix(matname)
    if (row==0) {
        submat = source[., col]'  // note the prime
    }
    else if (col==0) {
        submat = source[row, .]
    }
    else {
        submat = source[row, col]
    }
    
    st_local("matastring", invtokens(strofreal(submat)))
}

end




program define _dsimih_create, eclass sortpreserve
    
    version 11.2
    
    syntax ,                ///
         Step(numlist min=1 max=1 int >=0)  /// not optional; dsimih_create.ado passes 8 as default
       [ REGimes(numlist int sort >=0)      /// default is all regimes
         noSE               ///
         BS                 ///
         BSP                ///
         Reps(numlist min=1 max=1 >=3 int)    ///
         BSAving(string)    ///   filename, replace option, file confirm, etc already done
         BSBMat(name)       ///
         noDots             ///
         SAVing(string)     ///   filename, replace option, file confirm, etc already done
         noMAXopts          ///
         fromb              ///
         Verbose            ///
         noTable            ///
         replace            ///
         repnum(numlist min=1 max=1 int >0)]  //  undocumented; crucial internal option: indicates a recursive call from the bootstrap loop
                                              //  repnum is the bs repnum ; _dsimih_main() calcs the dsims and stores result in T_dsimih_bs[repnum]

    /*
    local `regimes' in -dsimih- always refers to the contents of option -regimes()-
    in -svarih-, `regimes' are all regimes in the model ; this is `rgms_model' in -dsimih-
    in both -svarih- and -dsimih, `numregimes' refers to all regimes in the model
       -dsimih create- uses `rgms_calc' for the regimes to be calculated (which may be different from option -regimes()-)
       and `numrgms_calc' or `num_s' (Mata) for the # of rgms to be calculated
    */

    local mshort = lower(substr("`e(method)'",1,3))

    local rgms_model    `e(regimes)'
    local numrgms_model `e(numregimes)'

    local rgmvar      = e(rgmvar)
    local depvar      "`e(depvar)'"
    local neqs        : word count `depvar'
    local mlag        = e(mlag)
    local lags        "`e(lags)'"

    local tminchar    = e(tmin)
    local tmaxchar    = e(tmax)
    local tsfmtchar    `e(tsfmt)'
    local timevarchar  `e(timevar)'

    if "`e(exog)'" != "" {
        local exog     "`e(exog)'"
        local exogopt  "exog(`exog')"
    }   
    else {
        local exog none
    }
    
    // E(SMAMPLE) AND TSSET
    if "`bs'`bsp'"!="" {
        qui count if e(sample)==1
        if r(N) == 0 {
            di as err "e(sample) is never equal to 1"
            exit 498
        }   
        if r(N) != e(N) {
            di as err "e(sample) inconsistent with # of obs recorded in e(N)"
            exit 471
        }
        
        qui _ts , sort onepanel
    }
    
    if "`repnum'"=="" {  // true if nose or seasmp was requested , or in main call for options bs or bsp
                         // repnum!="" if _dsimih_create calls itself recursively from within the bs/bsp bootstrap loop
        if `"`bsaving'"' != "" & "`bs'`bsp'" == "" {
            di as error "Options bs or bsp must be specified with bsaving()"
            exit 198
        }
    
        if `: word count `bs' `bsp' `se'' > 1 {
            di as error "Options nose, bs and bsp are mutually exclusive alternatives."
            exit 198
        }

        if "`reps'" != "" & "`bs'`bsp'"=="" {
            di as err "Option reps() cannot be specified without bs or bsp"
            exit 198    
        }

        if "`bs'`bsp'"!="" & "`reps'"=="" {
            disp as error `"Option reps() requires one of the options bs or bsp"'
            exit 198
        }
        
        if "`bsbmat'"!="" {
            if "`bs'`bsp'"=="" {
                disp as error `"Option bsbmat() requires one of the options bs or bsp."'
                exit 198
            }
            capture matrix drop `bsbmat' // delete matrix specified in all cases, even if -dsimih create- aborts with an error
                                         //   or if bsbmat is not calculated due to matsize
            if c(matsize)<`reps' {
                disp as text `"Omitting calculation of matrix `bsbmat' from option bsbmat() due to insufficient matsize setting"'.
                local bsbmat ""
            }
            tempname mybsb        // copy matrix contents of `mybsb' into `bsbmat' only for successful -dsimih create- runs
        }
    }
    
    /*
    local names to calculate sets of regimes for:
      > regimes to be newly calculated
      > matsize issues
      > regimes to be removed/added from the bootstrap repinfo matrix
    
    DSIM CALCULATIONS
    rgms_model          all rgms in model
    regimes             requested by user in option -regimes-
    rgms_exist          already in e()
    rgms_ovlp           overlap: = regimes & rgms_exist
    rgms_new            = regimes - rgms_exist
    rgms_ovlp_calc      rgms requested that exist but have step < `step'
    rgms_ovlp_nocalc    rgms requested but already exist with at least step `step'
    rgms_calc           = rgms_ovlp_calc + rgms_new
    
    MATSIZE
    rgms_exist_nocalc   = rgms_exist - rgms_ovlp_calc
    
    REPINFO MATRIX
    rgms_bs_exist       rgms in e(dsimih_bs/bsp), if it exists
    rgms_bs_nocalc      = rgms_bs_exist - rgms_calc
    rgms_convmat        rgms in new bootstrap repinfo matrix
    */
    
    
    // CHECK IF STATISTICS ALREADY EXIST, SAVE EXISTING STATS AS A FILE
    // the newly calculated stats are merged into the existing ones (existing ones will be the master data set)
    
    if "`repnum'"!="" {
        local rgms_calc `regimes'
    }
    else {
        if "`regimes'"=="" {
            local regimes `rgms_model'
        }
        else {
            local notin : list regimes - rgms_model
            if "`notin'"!="" {
                disp as error `"Regimes requested but not in estimation sample: `notin'"'
                exit 125
            }
        }

        capture _dsimih_chke  // returns _rc==9 if e() does not contain dsimih results
        if _rc {
            local rgms_calc `regimes'
        }
        else {
            preserve

            qui dsimih_describe , step
            local rgms_exist `r(stepmat_regimes)'
            local rgms_ovlp   : list regimes & rgms_exist  // "ovlp": overlap
            local rgms_new    : list regimes - rgms_exist

            tempname stepmat descrow
            matrix `stepmat' = r(stepmat)

            local rgms_ovlp_calc ""
            dsimih_use , clear noci se
            foreach rgm of local rgms_ovlp {
                local calc false
                local h_exist = `stepmat'[rownumb(`stepmat', "_`rgm'"), colnumb(`stepmat', "stat")]
                if "`h_exist'"=="." | `h_exist' < `step' {
                    qui replace sirf  = . if regime==`rgm'
                    qui replace sfevd = . if regime==`rgm'
                    local calc true
                }
                if "`bs'"!="" {
                    local h_exist = `stepmat'[rownumb(`stepmat', "_`rgm'"), colnumb(`stepmat', "sebs")]
                    if "`h_exist'"=="." | `h_exist' < `step' {
                        qui replace sirf_sebs   = . if regime==`rgm'
                        qui replace sfevd_sebs  = . if regime==`rgm'
                        local calc true
                    }
                }
                else if "`bsp'"!="" {
                    local h_exist = `stepmat'[rownumb(`stepmat', "_`rgm'"), colnumb(`stepmat', "sebsp")]
                    if "`h_exist'"=="." | `h_exist' < `step' {
                        qui replace sirf_sebsp  = . if regime==`rgm'
                        qui replace sfevd_sebsp = . if regime==`rgm'
                        local calc true
                    }
                }
                else if "`se'"!="nose" {
                    local h_exist = `stepmat'[rownumb(`stepmat', "_`rgm'"), colnumb(`stepmat', "seasmp")]
                    if "`h_exist'"=="." | `h_exist' < `step' {
                        qui replace sirf_seasmp  = . if regime==`rgm'
                        qui replace sfevd_seasmp = . if regime==`rgm'
                        local calc true
                    }
                }
                if "`calc'"=="true" local rgms_ovlp_calc `rgms_ovlp_calc' `rgm'
            }

            if "`rgms_ovlp_calc'`rgms_new'"=="" {
                disp as text `"All requested statistics already exist."'
                disp as text `"Use 'dsimih drop' first if you want to recalculate them."'
                exit
            }
            else {
                local rgms_ovlp_nocalc : list rgms_ovlp - rgms_ovlp_calc
                if "`rgms_ovlp_nocalc'"!="" {
                    disp as text `"Statistics for some regimes already exist and will not be re-calculated."'
                    disp as text `"Use 'dsimih drop' first if you want to recalculate them."'
                    disp as text `"Spells concerned: `rgms_ovlp_nocalc'"'
                }
                local rgms_calc  `rgms_ovlp_calc' `rgms_new'
                local rgms_calc : list sort rgms_calc
            }

            tempfile stats_exist
            qui save `stats_exist'
            
            restore
        }

        // CHECK MATSIZE
        local rgms_exist_nocalc : list rgms_exist - rgms_ovlp_calc
        local size = 0
        foreach rgm of local rgms_exist_nocalc {
            local hbar_rgm = `stepmat'[rownumb(`stepmat', "_`rgm'"), colnumb(`stepmat', "stat")]
            local size = `size' + `neqs'^2 * `= `hbar_rgm' + 1'
        }
        local size = `size' + `neqs'^2 * `=`step'+1'  * `: word count `rgms_calc''
        
        if `size'>c(matsize) {
            local noesave noesave
            if `"`saving'"'=="" {
                disp as error `"Your current matsize setting is `c(matsize)' but you need `size' for the current {cmd:dsimih} command."'
                disp as error `"You need to increase matsize in order to store {cmd:dsimih} results in {cmd:e()}."'
                disp as error `"See {help matsize}."'
                disp as error `"If your Stata flavor does not allow a sufficient matsize setting, you can use option {cmd:saving}"'
                disp as error `"to save results as a data set. See {help limits}."'
                exit 908
            }
            else {
                disp as text `"Note: Results will be available in the saved data set but not in {cmd:e()} due to the current matsize setting."'
            }
        }

        // PREPARE BOOTSTRAP RESULTS COUNT MATRICES
        if "`bs'`bsp'"!="" {
            tempname bsconvmat bsconvmat_exist jnk

            _dsimih_bsmatexist // returns locals `bsmat'="bsmat" and `bspmat'="bspmat" if they exist
            
            if "``bs'`bsp'mat'"!="" {
                matrix `bsconvmat_exist' = e(dsimih_`bs'`bsp')
                qui mat2mac `bsconvmat_exist' , col(1)
                local rgms_bs_exist `r(mat2mac)'
                local rgms_bs_nocalc : list rgms_bs_exist - rgms_calc
            }
            
            local rgms_convmat     `rgms_bs_nocalc' `rgms_calc'
            local rgms_convmat    : list sort rgms_convmat
            local numrgms_convmat : word count `rgms_convmat'

            foreach rgm in `rgms_convmat' {
                capture confirm matrix `bsconvmat_exist'
                if !_rc & !`: list rgm in rgms_calc' {
                    matrix `bsconvmat' = ( nullmat(`bsconvmat') \ `bsconvmat_exist'[rownumb(`bsconvmat_exist', "_`rgm'"), 1...] )
                }
                else {
                    matrix `jnk' = J(1, 6, .)
                    matrix rownames `jnk' = _`rgm'
                    matrix `bsconvmat' = ( nullmat(`bsconvmat') \ `jnk' )  // numbers will be filled in when bootstrap is finished
                }
            }
            matrix colnames `bsconvmat' = regime reps conv fail nonconv notident
        }
    }

    // BOOTSTRAP PREPARATION
    if "`reps'"!="" {                // bootstrap code starts here ----------------------------------

        if "`mshort'"=="bac" local mattypes a b e
        if "`mshort'"=="llu" local mattypes b l
        if "`mshort'"=="bfa" {
            local mattypes b e2
            if `numrgms_model'==3 local mattypes b e2 e3
            if `numrgms_model'==4 local mattypes b e2 e3 e4
            
            numlist "2/`numrgms_model'"
            local rgms_bfa_e `r(numlist)'
        }
        
        foreach curtype in `mattypes' {
            local cns_`curtype' "`e(cns_`curtype')'"

            macro drop T_cns_`curtype'_n
            while "`cns_`curtype''" != "" {
                gettoken next cns_`curtype':cns_`curtype' , parse(":")
                if "`next'" != ":" {
                    constraint free 
                    constraint define `r(free)' `next'
                    global T_cns_`curtype'_n ${T_cns_`curtype'_n} `r(free)'
                }
            }
            local `curtype'constraints "`curtype'constraints(${T_cns_`curtype'_n})"
        }
        
        tempname b sample b_varsim
        matrix `b'=e(b_var)
        
        if "`e(dfk_var)'" != "" local dfk = "dfk"

        qui gen byte `sample'=e(sample)
        local lagsopt "lags(`e(lags)')"
        
        local nocons "`e(nocons)'"  // `nocons'=="nocons" if it was used and recorded in e(), "" otherwise
        
        tempname vres
        tempvar  vres_smp
        
        // GEN DEPVARS AND SCORING MATRIX FOR VARSIM
        // depvar_varsim is the new depvar varlist, b_varsim is the e(b_var) w/ correspondingly modified colnames
        tempname b_varsim
        
        if inlist("`mshort'", "bac", "llu") {
            capture confirm matrix e(b_vargls)
            if _rc {
                matrix `b_varsim' = e(b_var)
            }
            else {
                matrix `b_varsim' = e(b_vargls)
            }
        }
        else if "`mshort'"=="bfa" {
            matrix `b_varsim' = e(b_var1)
            foreach rgm of local rgms_bfa_e {
                tempname b_rgm`rgm'
                matrix `b_rgm`rgm'' = e(b_var`rgm')
            }
        }
        local cnames `"`: colfullnames `b_varsim'' "' // note the space at the end ; translations have to be done with a space at the end
                                                      // otherwise errors would occur e.g. for endog vars "fx fx2"
        local i 1
        foreach curendog of local depvar {
            tempvar depvar_varsim`i'
            qui gen double `depvar_varsim`i'' = `curendog'
            local depvar_varsim "`depvar_varsim' `depvar_varsim`i''"
            local cnames : subinstr local cnames ".`curendog' " ".`depvar_varsim`i'' " , all
            local ++i
        }

        matrix colnames `b_varsim' = `cnames'
        if "`mshort'"=="bfa" {
            foreach rgm of local rgms_bfa_e {
                matrix colnames `b_rgm`rgm'' = `cnames'
            }
        }

        // PREPARING CMDLINE FOR BOOTSTRAP RUNS
        if "`e(evalmode)'"!="" local evalmode evalmode(`=substr("`e(evalmode)'",2,1)')

        local maxoptspec ""
        if "`maxopts'"!="nomaxopts" local mloptspec `"`e(mlopts)'"'

        local fromspec ""
        if "`fromb'"!="" {
            tempname bmat
            matrix `bmat' = e(b)
            local fromspec from(`bmat')
        }
        else if "`maxopts'"!="nomaxopts" {
            if `"`e(from)'"'=="matrix" {  // -from- was specified, using a matrix
                tempname frommat
                matrix `frommat' = e(from)
                local fromspec from(`frommat')
            }
            else if `"`e(from)'"'!="" { // -from- was specified, as a string
                local fromspec from(`e(from)')
            }
        }

        local genopts rgmvar(`rgmvar') ///
                     `lagsopt'         ///
                     `exogopt'         ///
                                       ///
                     `nocons'          ///
                     `dfk'             ///
                     `e(small)'        ///
                                       ///
                     `evalmode'        ///
                     `mloptspec'       ///
                     `fromspec'

        local noidencheck ""
        if inlist("`mshort'", "bac", "bfa") {
            if "`e(idencheck)'"=="skipped" local noidencheck noidencheck
        }

        if "`mshort'"=="bac" {
            tempname rgmmat_used
            matrix `rgmmat_used' = e(rgmmat_used)
            
            local bs_cmdline svarih bac `depvar' if `sample' , `genopts' ///
                     rgmmat(`rgmmat_used')   ///
                    `aconstraints'           ///
                    `bconstraints'           ///
                    `econstraints'           ///
                    `noidencheck'            ///
                    `e(glsopts)'             //
        }
        else if "`mshort'"=="bfa" {
            local bs_cmdline svarih bfa `depvar' if `sample' , `genopts' ///
                    `bconstraints'           ///
                    `e2constraints'          ///
                    `noidencheck'            //
            if `numrgms_model'==3 local bs_cmdline `"`bs_cmdline' `e3constraints'"'
            if `numrgms_model'==4 local bs_cmdline `"`bs_cmdline' `e3constraints' `e4constraints'"'
        }
        else if "`mshort'"=="llu" {
            local bs_cmdline svarih llu `depvar' if `sample' , `genopts' ///
                    `bconstraints'           ///
                    `lconstraints'           ///
                    `e(glsopts)'             //
        }

        mata: T_dsimih_bs = J(`reps',1,NULL)  // advantage of this approach over an external declaration in _dsimih_main()
                                              //   is that T_dsimih_bs gets reset if a previous -dsimih- run failed to clean up globals

        // BOOTSTRAP
        local noi ""
        if "`verbose'"=="" & "`dots'"!="nodots" {
            disp as text _n "Bootstrap replications ({bf:`reps'})"
            local progline "{c -}{c -}{c -}{c +}{c -}{c -}{c -}"
            local progline "{c -}`progline' 1 `progline' 2 `progline' 3 `progline' 4 `progline' 5"
            disp as text "`progline'"
        }
        else {
            local noi noi
        }

        local conv_cnt     0  // converged results are never a 'fail'
        local fail_cnt     0
        local nonconv_cnt  0  // non-converged results are mostly not fails
        local notident_cnt 0  // estimates did not pass local identification check
        if "`mshort'"=="llu" | "`noidencheck'"=="noidencheck" local notident_cnt "."

        forvalues r=1/`reps' {

            qui preserve

            if "`verbose'"!="" disp as txt _n "{hline}" _n "BOOTSTRAP SAMPLE #`r'"

            if inlist("`mshort'", "bac", "llu") {
                _pub__varsim `depvar_varsim' , neqs(`neqs') b(`b_varsim')                 `bsp' fsamp(`sample') eqlist(`depvar') rgmvar(`rgmvar')
            }
            else if "`mshort'"=="bfa" {
                if `numrgms_model'==3 local brgmopt brgm3(\`b_rgm3')
                if `numrgms_model'==4 local brgmopt brgm3(\`b_rgm3') brgm4(\`b_rgm4')
                _pub__varsim `depvar_varsim' , neqs(`neqs') b(`b_varsim') brgm2(`b_rgm2') `bsp' fsamp(`sample') eqlist(`depvar') rgmvar(`rgmvar')  `brgmopt'
            }

            forvalues i = 1/`neqs' {
                local tv1 : word `i' of `depvar_varsim'
                local tv2 : word `i' of `depvar'
                qui replace `tv2' = `tv1'
            }

            _est hold `vres', copy restore nullok varname(`vres_smp') 

            *********************************
            capture `noi' `bs_cmdline' `table'
* noi disp `"`bs_cmdline'"'
            *********************************

            local repchar ""
            if "`e(idencheck)'"=="failed" {
                local repchar i
                local ++notident_cnt
            }
            else if _rc!=0 | e(rc_ml)!=0 {
                local repchar f
                local ++fail_cnt
            }
            else {  // all conditions on e-values should be quoted since results may not exist if the svarih command may have failed
                    // in these cases dsimih would abort with an error
                local converged 0
                if inlist("`mshort'", "bac", "bfa") {
                    if "`e(converged_ml)'"=="1" local converged 1
                }
                else if inlist("`mshort'", "llu"       ) {
                    if "`e(converged_ml)'"=="1"  & ( "`e(converged_gls)'"=="1" | "`e(glsiter)'"=="0" ) local converged 1
                }
                
                if `converged' {
                    
                    if "`bsbmat'"!="" {
                        matrix `mybsb' = ( nullmat(`mybsb') \ e(b) )  // colnames are automatically those of e(b)
                        local  bsbmat_rn `bsbmat_rn' rep`r'
                    }
                    
                    local repchar "."
                    local ++conv_cnt

                    ***********************************************************************
                    qui _dsimih_create , regimes(`rgms_calc') step(`step') nose repnum(`r')
                    ***********************************************************************

                    // note: this call has very little overhead
                    // each call only assembles the basic model features from e(), but does not check the options input over and over again
                }
                else {
                    local repchar n
                    local ++nonconv_cnt
                }
            }

            local repchars `repchars' `repchar'  // needed in _dsimih_mergebs()

            if "`verbose'"=="" {
                if "`dots'" != "nodots" {
                    if inlist("`repchar'", "f", "n", "i") {
                        disp in red  "`repchar'" _c
                    }
                    else {
                        disp as text "." _c
                    }
                }   

                if mod(`r', 50)==0 {
                    disp as text _dup(`=8-length("`r'")') " " "`r'"
                }
                if `r' == `reps' {
                    disp as text "" 
                }
            }
            else {
                disp ""
                disp as text "converged_ml : `e(converged_ml)'"
                if inlist("`mshort'", "bac", "llu") {
                    if "`e(glsiter)'"!="0" disp as text "converged_gls: `e(converged_gls)'"
                }
            }

            _est unhold `vres'  // removed -capture- so checking for user interrupt works

            qui restore

            if _rc==1 qui exit 1
        }
    }
    
    // --------------------- bootstrap code ends here ---------------------------------------------------------
    
    tempname svarA svarAI
    if "`mshort'"=="bac" {
        matrix `svarA'  = e(A)
        capture matrix `svarAI' = inv(`svarA')
        if _rc > 0 {
            di as err "svar estimate of A matrix is not invertible"
            di as err "cannot estimate structural IRFs"
            exit 498
        }
    }
    else {  // bfa and llu have A=I(neqs)
        matrix `svarA'  = I(`neqs')
        matrix `svarAI' = I(`neqs')
    }
    
    if "`repnum'"!="" {
        mata: _dsimih_main("nose", `repnum')
        exit
    }
    
    if "`bs'`bsp'"!="" local se nose
    if "`se'" != "nose" local se se

    mata: T_dsimih = J(1,1,NULL) // see note above about this method of declaring/setting a global
    
    if "`se'"!="nose" & "`dots'"!="nodots" {  // TODO: draw this in mata right before se calcs, since calc of stats may take a long time
        disp as text _n "Calculating asymptotic standard errors ({bf:step 0-`step'})"
        local progline "{c -}{c -}{c -}{c +}{c -}{c -}{c -}"
        local progline "{c -}`progline' 1 `progline' 2 `progline' 3 `progline' 4 `progline' 5"
        disp as text "`progline'"
    }
    
    mata: _dsimih_main("`se'", 0)

    if "`bs'`bsp'"!="" {
        
        if `conv_cnt'<3 {
            disp as error `"could not obtain more than 2 valid boostrap estimates"'
            disp as error `"after `reps' replications"'
            exit 430
        }
        
        mata: _dsimih_mergebs()

        foreach rgm of local rgms_calc {
            matrix `bsconvmat'[rownumb(`bsconvmat', "_`rgm'"), 1] = `rgm'
            matrix `bsconvmat'[rownumb(`bsconvmat', "_`rgm'"), 2] = `reps'
            matrix `bsconvmat'[rownumb(`bsconvmat', "_`rgm'"), 3] = `conv_cnt'
            matrix `bsconvmat'[rownumb(`bsconvmat', "_`rgm'"), 4] = `fail_cnt'
            matrix `bsconvmat'[rownumb(`bsconvmat', "_`rgm'"), 5] = `nonconv_cnt'
            matrix `bsconvmat'[rownumb(`bsconvmat', "_`rgm'"), 6] = `notident_cnt'
        }

        if `"`bsaving'"'!="" {
            preserve
            if "`bs'`bsp'"!="" {
                dsimih_use , noci se mata(`bs'`bsp') bsreps clear
                qui drop sirf_se sfevd_se
            }
            save `bsaving'
            restore
        }
    }

    if "`saving'"!="" {
        tempname stepmat
        preserve
        if "`bs'`bsp'"!="" {
            dsimih_use , noci se mata(`bs'`bsp') clear
            _dsimih_getstepmat
            matrix `stepmat' = r(stepmat)
            _dsimih_addchars , stepmat(`stepmat') repmat(`bsconvmat') `bs' `bsp'
        }
        else {
            dsimih_use , noci se mata(asmp) clear
            _dsimih_getstepmat
            matrix `stepmat' = r(stepmat)
            _dsimih_addchars , stepmat(`stepmat')
        }
        save `saving'
        restore
    }

    if "`noesave'"=="" {
        tempname newmat
        mata: st_matrix("`newmat'", **(findexternal("T_dsimih")))

        local numrows = rowsof(`newmat')
        if "`bs'"!="" {
            matrix `newmat' = `newmat'[1..., 1..6] , J(`numrows', 2, .) , `newmat'[1..., 7..8] , J(`numrows', 2, .)
        }
        else if "`bsp'"!="" {
            matrix `newmat' = `newmat'[1..., 1..6] , J(`numrows', 4, .) , `newmat'[1..., 7..8]
        }
        else if "`se'"=="nose" {
            matrix `newmat' = `newmat'[1..., 1..6] , J(`numrows', 6, .)
        }
        else {
            matrix `newmat' = `newmat'[1..., 1..8] , J(`numrows', 4, .)
        }
        matrix colnames `newmat' = regime step imp_idx resp_idx sirf sfevd sirf_seasmp sfevd_seasmp sirf_sebs sfevd_sebs sirf_sebsp sfevd_sebsp

        capture confirm matrix e(dsimih)
        if !_rc {
            if colsof(e(dsimih))>1 {  // 'dsimih drop , erase' stores e(dsimih)=J(1,1,.)
                tempname est
                _estimates hold `est' , copy restore
                ereturn matrix dsimih = `newmat'
                preserve
                dsimih_use , clear se noci
                qui merge 1:1 regime step impulse response using `"`stats_exist'"', update nogen noreport

                _dsimih_tostore, depvar(`depvar') matname(`newmat')

                restore
                _estimates unhold `est'
            }
        }
        
        if "`bs'" !="" ereturn matrix dsimih_bs  = `bsconvmat' , copy  // `bsconvmat' still needed for -saving()- option
        if "`bsp'"!="" ereturn matrix dsimih_bsp = `bsconvmat' , copy
        ereturn matrix dsimih = `newmat'
/*
        if "`saving'"!="" {
            tempname stepmat
            preserve
            dsimih_use , noci se clear
            _dsimih_getstepmat
            matrix `stepmat' = r(stepmat)
            _dsimih_addchars , stepmat(`stepmat') repmat(`bsconvmat') `bs' `bsp'
            save `saving'
            restore
        }
*/
    }
    else {
/*
        if "`saving'"!="" {
            tempname stepmat
            preserve
            if "`bs'`bsp'"!="" {
                dsimih_use , noci se mata(`bs'`bsp') clear
                _dsimih_getstepmat
                matrix `stepmat' = r(stepmat)
                _dsimih_addchars , stepmat(`stepmat') repmat(`bsconvmat') `bs' `bsp'
            }
            else {
                dsimih_use , noci se mata(asmp) clear
                _dsimih_getstepmat
                matrix `stepmat' = r(stepmat)
                _dsimih_addchars , stepmat(`stepmat')
            }
            save `saving'
            restore
        }
*/
    }
    
    if "`bsbmat'"!="" {
        matrix `bsbmat' = `mybsb'
        matrix rownames `bsbmat' = `bsbmat_rn'
    }        

end

*** --------------------------------- MATA ------------------------------------------------

version 11.2
mata:
mata set matastrict on
    void _dsimih_main(string scalar se, real scalar repnum) {
    // calculates SIRFs and SFEVDs
    // arg se=="se"/"nose" indicates whether it also calcs the s.e.

        string scalar mshort, dots
        
        string rowvector names, rmvars
        
        real scalar i, j, k, s, h, neqs, nn, nl, nnl,  //
                    maxlag, numlags, num_s, hbar, cnt, // num_s: =numrgms_calc
                    hidx, iidx, jidx, kidx, cntidx,    // h as it is used in Mata vectors (which are base-1, but indexes often start at period 0)
                    nvl                                // "nvl": numvarloop, equal to numspells for bfa, 1 otherwise ; controls # of loops for var-related concepts
        
        real rowvector laglist, rgms_calc, nc, lags
        
        real colvector order, nr
        
        real vector idx, idx2
        
        real matrix b, V, ordermat, sigmaPi, M, Mtr, Jl, A, Ainv,
                    Fbar, Fbarinv, Mbar, P, Pinv, Binv, Qbar, sigma0, Q6fix, Q9fix,
                    Q6, Q9, jnk, sirf, sirf_se, sfevd, sfevd_se, outmat

        pointer(real matrix) colvector pA,
                                       pM,         // powers of companion matrix
                                       pMtr,       //   "tr": transpose
                                       pPhi,
                                       pBs,         // = B+E*Dt(s)
                                       pSigmaABs,
                                       pGbar,
                                       pIJMJ,
                                       pZbar,
                                       pPJM

        pointer(real matrix) matrix    pRgmBV,   // contains [pBs, pSigmaABs]
                                       pIJMJsigma0,
                                       pSigh,       // (h+1) x (h+1) (block-wise defined matrix)
                                       pSirf,       // (h+1) x s
                                       pSirf_se,    // (h+1) x s
                                       pSfevd,      // (h+1) x s
                                       pSfevd_se    // (h+1) x s

        pointer(real matrix)       colvector pSigmaPi, pb, pV        // related to VAR-concepts that change for bfa
        pointer(pointer colvector) colvector ppPhi, ppM, ppMtr, ppA


        if (!any(se:==("se","nose"))) _error(198)
        
        mshort = st_local("mshort")
        dots   = st_local("dots")

        rgms_calc = strtoreal(tokens(st_local("rgms_calc")))
        num_s   = cols(rgms_calc)
        
        neqs    = strtoreal(st_local("neqs"))
        lags    = strtoreal(tokens(st_global("e(lags)")))  // contains expanded numlist
        numlags = cols(lags)
        maxlag  = lags[numlags]

        hbar = strtoreal(st_local("step"))

        nn  = neqs*neqs
        nl  = neqs*numlags       // later redefined if maxlag>numlags
        nnl = neqs*neqs*numlags  // later redefined if maxlag>numlags

        if (mshort!="bfa") {  // VAR-related concepts vary for bfa but not for bac, llu
            nvl = 1
        }
        else {
            nvl = strtoreal(st_local("numrgms_model"))
        }
        pb        = J(nvl,1,NULL)
        pV        = J(nvl,1,NULL)
        ppPhi     = J(nvl,1,NULL)
        pSigmaPi  = J(nvl,1,NULL)
        ppM       = J(nvl,1,NULL)
        ppMtr     = J(nvl,1,NULL)
        ppA       = J(nvl,1,NULL)

        if (mshort!="bfa") {
            pb[1] = & (st_matrix("e(b_vargls)") )
            if (*(pb[1])==J(0,0,.)) {
                pb[1] = &( st_matrix("e(b_var)") )
                pV[1] = &( st_matrix("e(V_var)") )
                names = st_matrixcolstripe("e(b_var)")
            }
            else {
                pV[1] = &( st_matrix("e(V_vargls)") )
                names = st_matrixcolstripe("e(b_vargls)")
            }
        }
        else {
            names = st_matrixcolstripe("e(b_var1)")
            for (i=1;i<=nvl;i++) {
                pb[i] = &( st_matrix("e(b_var" + strofreal(i) + ")") )
                pV[i] = &( st_matrix("e(V_var" + strofreal(i) + ")") )
            }
        }

        // remove exogvars and constant from b, V; afterwards reorder elements
        rmvars = (tokens(st_local("exog")) , "_cons")

        idx = J(cols(*pb[1]), 1, 0)
        for (i=1; i<=cols(rmvars); i++) {
            idx = idx :| (names[.,2]:==rmvars[i])
        }

        // conversion to vec(A1,...,Ap)
        // indexes in e(b) are from fastest to slowest: lagnum-regressor-eqname
        // convert this to vec(A1,...,Ap) form        : eqname-regressor-lagnum
        ordermat =  ( (1::neqs) # J(nl,1,1)                    ,   // eqname
                      J(neqs,1,1) # (1::neqs) # J(numlags,1,1) ,   // regressor
                      J(nn,1,1) # (1::numlags)                 ,   // lagnum
                      (1::nnl)                                   ) // idx: position of elems in e(b_var) as returned from -svarih-

        _sort(ordermat, (3,2,1))  // now lagnum slowest, ..., eqname fastest
        order = ordermat[.,4]

        // fill in zeros if some lag matrices have been excluded in estimation
        // idx2 identifies the row/col pos that the elems of b/V should receive in full bf/Vf
        // e.g. for lags(1 4) and neqs=2, generates (0,0,0,0,12,12,12,12) + (1,2,3,4,1,2,3,4)
        idx2 = ((neqs^2*(lags:-1)) # J(1,neqs^2,1)) + (J(1,numlags,1) # (1..neqs^2))

        for (i=1;i<=nvl;i++) {

            // remove exogvars, constants
            pb[i] = &( select(*pb[i],!idx') )
            pV[i] = &( select(*pV[i],!idx)  ) // delete rows
            pV[i] = &( select(*pV[i],!idx') ) // delete cols

            // conversion to vec(A1,...,Ap)
            pb[i] = &( (*pb[i])[order'] )         // now b=vec(A1,...,Ap)'
            pV[i] = &( (*pV[i])[order, order'] )

            // fill in zeros
            if (numlags<maxlag) {
                jnk = J(1, nn*maxlag, 0)
                jnk[1, idx2] = *pb[i]
                pb[i] = &( 1*jnk )
                jnk = J(nn*maxlag, nn*maxlag, 0)
                jnk[idx2,idx2] = *pV[i]
                pV[i] = &( 1*jnk )
            }

            pb[i] = &( rowshape(*pb[i], maxlag) ) // now b=(vec(A1)' \ ... \ vec(Ap)')
        }
        nl  = neqs * maxlag
        nnl = neqs * neqs * maxlag

        pSigmaPi = pV

        // powers of companion matrix
        for (i=1;i<=nvl;i++) {
            ppA[i] = &( J(maxlag,1,NULL) )
            for (j=1;j<=maxlag;j++) {
                (*ppA[i])[j] = &(colshape((*pb[i])[j,.],neqs)')
            }
        }

        for (i=1;i<=nvl;i++) {
            ppM[i] = &( J(hbar+1,1,NULL) )
            
            M = J(neqs, nl, .)
            pA = *ppA[i]
            for (j=1;j<=maxlag;j++) {
                M[|1,(j-1)*neqs+1 \ neqs, j*neqs|] = *pA[j]
            }
            M = M \ ( I((maxlag-1)*neqs) , J((maxlag-1)*neqs, neqs, 0) )
            (*ppM[i])[1]     = &(I(nl))
            if (hbar>0) {
                (*ppM[i])[2] = &(1*M)
            }
            for (h=3;h<=hbar+1;h++) {
                (*ppM[i])[h]   = &( M   * (*(*ppM[i])[h-1]) )

            }
        }

        // powers of M' are transpose of powers of M since (A^2)'=(A*A)'=A'*A'=A'^2
        for (i=1;i<=nvl;i++) {
            ppMtr[i] = &( J(hbar+1,1,NULL) )
            for (h=1;h<=hbar+1;h++) {
                (*ppMtr[i])[h]   = &( ((*(*ppM[i])[h])')[1::neqs, .] )    // only first block row of Mtr is needed
                (*ppM[i])[h]     = &( (*(*ppM[i])[h])[1::neqs, 1..neqs] ) // only top left block of M is needed
            }
        }

        // VMA
        for (i=1;i<=nvl;i++) {
            pA = *ppA[i]
            ppPhi[i] = &( J(hbar+1, 1, NULL) )
            (*ppPhi[i])[1] = &(I(neqs))

            for (h=1;h<=hbar;h++) {
                hidx = h + 1
                for (j=1;j<=h;j++) {
                    if (j<=maxlag) {
                        if (j==1) {
                            (*ppPhi[i])[hidx] = &(                 *(*ppPhi[i])[hidx-j] * (*pA[1]) )
                        }
                        else {
                            (*ppPhi[i])[hidx] = &( *(*ppPhi[i])[hidx] + *(*ppPhi[i])[hidx-j] * *pA[j] )
                        }
                    }
                }
            }
        }

        // regime-specific lincoms
        if (mshort=="bac") {
            pRgmBV = _dsimih_bac_mklincom(rgms_calc)
            A = st_matrix("e(A)")
            Ainv = luinv(A)
        }
        else if (mshort=="bfa") {
            pRgmBV = _dsimih_bfa_mklincom(rgms_calc)
            A    = I(neqs)
            Ainv = I(neqs)
        }
        else if (mshort=="llu") {
            pRgmBV = _dsimih_llu_mklincom(rgms_calc)
            A    = I(neqs)
            Ainv = I(neqs)
        }

        pBs       = pRgmBV[.,1]
        pSigmaABs = pRgmBV[.,2]

        Jl = (I(neqs) , J(neqs, neqs*(maxlag-1), 0))  // "Jl": J large
        pragma unused Jl  // replaced Jl mutliplications by matrix indexing but left it in the code for debugging

        pGbar = J(hbar+1,1,NULL)
        pIJMJ = J(hbar+1,1,NULL)
        pZbar = J(hbar ,1,NULL)
        pSigh = J(hbar+1,hbar+1,NULL)
        
        pSirf     = J(hbar+1,num_s,NULL)
        pSirf_se  = J(hbar+1,num_s,NULL)
        pSfevd    = J(hbar+1,num_s,NULL)
        pSfevd_se = J(hbar+1,num_s,NULL)

        pGbar[1] = &(J(nn,nnl,0))
        pIJMJ[1] = &(I(neqs^2))
        pGbar[1] = &(J(nn,nnl,0))

        nr = 1::neqs
        nc = 1..neqs
        pPJM = J(hbar,1,NULL)
        pIJMJsigma0 = J(hbar+1,hbar+1,NULL)

        pPhi     = *ppPhi[1]
        sigmaPi  = *pSigmaPi[1]
        pM       = *ppM[1]
        pMtr     = *ppMtr[1]
        
        for (s=1;s<=num_s;s++) {
            if (mshort=="bfa" & s>1) {
                pPhi     = *ppPhi[s]
                sigmaPi  = *pSigmaPi[s]
                pM       = *ppM[s]
                pMtr     = *ppMtr[s]
            }
            pSfevd[1,s] = &(J(neqs, neqs, 0))

            Fbar = J(neqs, neqs, 0)
            Mbar = J(neqs, neqs, 0)
            P = Ainv * (*pBs[s])
            Pinv = luinv(P)
            Binv = luinv(*pBs[s])

            if (se=="se") {

                // defining terms to optimize nested loops for hbar
                Qbar   = (P' # P) * ((I(neqs) # Binv ) , -(Pinv' # Binv))
                sigma0 = Qbar*(*pSigmaABs[s])*Qbar'

                for (cnt=0;cnt<hbar;cnt++) {
                    cntidx = cnt + 1
                    pPJM[cntidx] = &( P'*(*pMtr[cntidx]) )
                }
                for (h=1;h<=hbar;h++) {
                    hidx = h+1
                    for (k=0;k<h;k++) {  // costly nested loop
                        kidx = k + 1
                        cnt = h-1-k
                        cntidx = cnt + 1
                        if (kidx==1) {
                            // pGbar[hidx] = &(                    (P'*Jl*(*pMtr[cntidx])) # (Jl*(*pM[kidx])*Jl')   )
                            pGbar[hidx] = &(                    (*(pPJM[cntidx])) # (*pM[kidx])   )
                                          // note: this is inefficient as the same kron products are calculated
                                          //       over and over again; storing all kron combs of PJM[cndidx] and M[kidx] in a pointer
                                          //       matrix is the alternative but, here memory becomes problematic
                                          //       e.g. neqs=6, 24 lags, hbar=100 this would require ~30MB
                        }
                        else {
                            // pGbar[hidx] = &( (*pGbar[hidx]) + ( (P'*Jl*(*pMtr[cntidx])) # (Jl*(*pM[kidx])*Jl') ) )
                            pGbar[hidx] = &( (*pGbar[hidx]) + ( (*(pPJM[cntidx])) # (*pM[kidx]) ) )
                        }
                    }

                }

                for (h=1;h<=hbar;h++) {
                    hidx = h + 1
                    pIJMJ[hidx] = &( I(neqs) # (*pM[hidx]) )
                }

                for (i=0;i<=hbar;i++) {
                    iidx = i + 1
                    for (j=0;j<=hbar;j++) {
                        jidx = j + 1
                        pIJMJsigma0[iidx,jidx] = &( (*pIJMJ[iidx]) * sigma0 * (*pIJMJ[jidx])' )
                    }
                }
            }

            for (h=0;h<=hbar;h++) {
                hidx = h+1
                pSirf[hidx,s] = &((*pPhi[hidx]) * P)
                if (h==0) {
                    if (se=="se") {
                        // pSirf_se[hidx,s] = &( diagonal( (*pIJMJ[hidx]) * sigma0 * (*pIJMJ[hidx])') )
                        pSirf_se[hidx,s] = &( diagonal( (*(pIJMJsigma0[hidx,hidx])) ) ) // the order is as in vec(A') !!
                        pSirf_se[hidx,s] = &( rowshape(*pSirf_se[hidx,s], neqs)' )
                        pSfevd_se[1,s] = &(J(neqs,neqs,0))
                    }
                }
                else {
                    if (se=="se") {
                        // pSirf_se[hidx,s] = &( diagonal((*pGbar[hidx]) * sigmaPi * (*pGbar[hidx])' + (*pIJMJ[hidx]) * sigma0 * (*pIJMJ[hidx])') )
                        pSirf_se[hidx,s] = &( diagonal( (*pGbar[hidx]) * sigmaPi * (*pGbar[hidx])' + (*(pIJMJsigma0[hidx,hidx])) ) )
                        pSirf_se[hidx,s] = &( rowshape(*pSirf_se[hidx,s], neqs)' )
                    }

                    Fbar = Fbar + (((*pSirf[hidx-1,s]) * (*pSirf[hidx-1,s])') :* I(neqs))
                    Fbarinv = luinv(Fbar)
                    Mbar = Mbar + (*pSirf[hidx-1,s]):^2
                    pSfevd[hidx,s] = &(Fbarinv * Mbar)

                    if (se=="se") {
                        Q6fix = I(neqs) # Fbarinv
                        Q9fix = ((*pSfevd[hidx,s])' # Fbarinv) * diag(vec(I(neqs))) * 0.5 * (I(neqs^2) + Kmatrix(neqs, neqs))

                        for (i=0;i<h;i++) {  // costly nested loop
                            iidx = i+1
                            Q6 = Q6fix * diag(vec(*pSirf[iidx,s]))
                            Q9 = Q9fix * (*pSirf[iidx,s] # I(neqs))
                            pZbar[iidx] = &( 2*(Q6-Q9) )
                        }

                        for (i=0;i<=h;i++) {  // costly nested loop
                            iidx = i+1
                            for (j=0;j<=h;j++) {
                                jidx = j+1
                                pSigh[iidx,jidx] = &( (*pGbar[iidx]) * sigmaPi * (*pGbar[jidx])' + (*pIJMJsigma0[iidx,jidx]) )
                            }
                        }

                        jnk = J(neqs^2, neqs^2, 0)
                        for (i=0;i<h;i++) {  // costly nested loop
                            iidx = i+1
                            for (j=0;j<h;j++) {
                                jidx = j+1
                                jnk = jnk + (*pZbar[iidx]) * (*pSigh[iidx,jidx]) * (*pZbar[jidx])'  // loop calculates the matrix product Zbar_h * Sigma_h * Zbar_h'
                            }
                        }
                        pSfevd_se[hidx,s] = &( diagonal(jnk) )
                        pSfevd_se[hidx,s] = &( rowshape(*pSfevd_se[hidx,s], neqs)' )
                    }                    
                }
                if (se=="se" & dots!="nodots" & s==num_s) {
                    printf(".")
                    if (mod(hidx,50)==0) {
                        display("")
                    }
                    displayflush()
                }
            }
        }
        display("")
       
        sirf     = J(neqs*neqs, (hbar+1)*num_s, .)
        sirf_se  = J(neqs*neqs, (hbar+1)*num_s, .)
        sfevd    = J(neqs*neqs, (hbar+1)*num_s, .)
        sfevd_se = J(neqs*neqs, (hbar+1)*num_s, .)

        j = 1
        for (s=1;s<=num_s;s++) {
            for (h=0;h<=hbar;h++) {
                hidx = h + 1
                sirf[.,j]     =      vec(*pSirf[hidx,s])
                sfevd[.,j]    =      vec(*pSfevd[hidx,s])
                if (se=="se") {
                    // sirf_se[.,j]  = sqrt(vec( (*pSirf_se[hidx,s])' ))
                    sirf_se[.,j]  = sqrt(vec(*pSirf_se[hidx,s]))
                    sfevd_se[.,j] = sqrt(vec(*pSfevd_se[hidx,s]))
                }
                j = j + 1
            }
        }
        // outmat columns: regime, step, impvar_idx, respvar_idx, sirf, sfevd, sirf_se, sfevd_se
        outmat   = J(num_s*(hbar+1)*neqs*neqs, 8, .)
        if (se=="se") {
            outmat[.,5..8] = (vec(sirf) , vec(sfevd) , vec(sirf_se) , vec(sfevd_se))
        }
        else {
            outmat[.,(5,6)] = (vec(sirf) , vec(sfevd))
        }

        // vec(sirf) is sorted:  regime step impulse response  (slowest to fastest moving index)
        outmat[., 1] = rgms_calc' # J((hbar+1)*neqs*neqs,1,1)
        outmat[., 2] = J(num_s,1,1) # ((0::hbar) # J(neqs*neqs,1,1))
        outmat[., 3] = J(num_s*(hbar+1),1,1) # ((1::neqs) # J(neqs,1,1))
        outmat[., 4] = J(num_s*(hbar+1)*neqs,1,1) # (1::neqs)

        // irf files are sorted: irfname step response impulse  ; -dsimih- sorts regime-impulse-response-step
        _sort(outmat, (1,3,4,2)) 

        if (repnum>0) {
            (*(findexternal("T_dsimih_bs")))[repnum] = &outmat  // TODO: I should not store the first four columns (regime-resp_idx) for each replication
                                                                //       if option -bsaving()- is not used
        }
        else {
            *(findexternal("T_dsimih")) = &outmat
        }
    }
end

version 11.2
mata:
mata set matastrict on
    pointer(real matrix) matrix _dsimih_bac_mklincom(real rowvector rgms_calc) {

        real scalar i, j,
                    neqs,
                    neqs2,
                    rgm
        
        real matrix rgmmat_used,
                    Dt,
                    lincom,
                    B,
                    E
        
        pointer(real matrix) matrix pRgmBV
        
        pRgmBV = J(cols(rgms_calc),2,NULL)

        B = st_matrix("e(B)")
        E = st_matrix("e(E)")
        neqs = cols(B)
        Dt = J(0,0,.)
        rgmmat_used = st_matrix("e(rgmmat_used)")
        for (j=1;j<=cols(rgms_calc);j++) {
            rgm = rgms_calc[j]
            for (i=1;i<=rows(rgmmat_used);i++) {
                if (rgmmat_used[i,1]==rgm) {
                    Dt = rgmmat_used[i,2..cols(rgmmat_used)]
                    break
                }
            }
            if (Dt==J(0,0,.)) {
                _error("_dsimih_bac_mklincom: regime " + strofreal(rgm) + "not in regime matrix?")
            }

            pRgmBV[j,1] = &(B + E*diag(Dt))

            neqs2 = neqs^2
            lincom = (I(neqs2)         , J(neqs2,neqs2,0) , J(neqs2,neqs2,0)     \
                      J(neqs2,neqs2,0) , I(neqs2)         , diag(Dt#J(1,neqs,1)) )
            pRgmBV[j,2] = &( lincom * st_matrix("e(V)") * lincom' )  // sigmaAB
        }
        return(pRgmBV)
    }
end

version 11.2
mata:
mata set matastrict on
    pointer(real matrix) matrix _dsimih_bfa_mklincom(real rowvector rgms_calc) {

        real scalar rgm, numregimes, neqs, neqs2
        
        real matrix B, V, lincom, sigmaB
        
        pointer(real matrix) matrix pRgmBV
        
        numregimes = st_numscalar("e(numregimes)")
        pRgmBV = J(numregimes,2,NULL)  // calcs are always done for all regimes in e(), but only a subset of regimes may be returned if requested

        B = st_matrix("e(B)")
        V = st_matrix("e(V)")
        neqs  = cols(B)
        neqs2 = neqs^2

        for (rgm=1;rgm<=numregimes;rgm++) {
            if (rgm==1) {
                pRgmBV[1,1] = &( 1*B )
                sigmaB        = 1* V[1::neqs2, 1..neqs2]
                pRgmBV[1,2] = &( J(neqs2, neqs2, 0) , J(neqs2, neqs2, 0) \ J(neqs2, neqs2, 0) , sigmaB )
            }
            else if (rgm==2) {
                pRgmBV[2,1] = &( B + st_matrix("e(E2)") )
                lincom = ( I(neqs2) , I(neqs2) , J(neqs2, neqs2*(numregimes-2), 0) )
                sigmaB = lincom * V * lincom'
                pRgmBV[2,2] = &( J(neqs2, neqs2, 0) , J(neqs2, neqs2, 0) \ J(neqs2, neqs2, 0) , sigmaB )
            }
            else if (rgm==3) {
                pRgmBV[3,1] = &( B + st_matrix("e(E3)") )
                lincom = ( I(neqs2) , J(neqs2, neqs2, 0), I(neqs2), J(neqs2, neqs2*(numregimes-3), 0) )
                sigmaB = lincom * V * lincom'
                pRgmBV[3,2] = &( J(neqs2, neqs2, 0) , J(neqs2, neqs2, 0) \ J(neqs2, neqs2, 0) , sigmaB )
            }
            else if (rgm==4) {
                pRgmBV[4,1] = &( B + st_matrix("e(E4)") )
                lincom = ( I(neqs2) , J(neqs2, 2*neqs2, 0), I(neqs2) )
                sigmaB = lincom * V * lincom'
                pRgmBV[4,2] = &( J(neqs2, neqs2, 0) , J(neqs2, neqs2, 0) \ J(neqs2, neqs2, 0) , sigmaB )
            }
        }

        // rowvector rgms_calc is sorted
        return(pRgmBV[rgms_calc,.])
    }
end

version 11.2
mata:
mata set matastrict on
    pointer(real matrix) matrix _dsimih_llu_mklincom(real rowvector rgms_calc) {
        
        real scalar i,
                    neqs,
                    neqs2
        
        real matrix L,
                    B,
                    Lstack,
                    Bstack,
                    lincom,
                    signame1,
                    signame2

        pointer(real matrix) matrix pRgmBV

        pRgmBV = J(2,2,NULL)   // calcs are always done for regimes 1 and 2, but only one of them may be returned if requested
        
        B = st_matrix("e(B)")
        L = st_matrix("e(L)")  // row vector
        neqs = cols(B)

        pRgmBV[1,1] = &( B )
        pRgmBV[2,1] = &( B * diag(L:^(0.5)) )
        
        neqs2 = neqs^2
        signame1 = st_matrix("e(V)")[1..neqs2, 1..neqs2]
        pRgmBV[1,2] = &( J(neqs2, neqs2, 0) , J(neqs2, neqs2, 0) \ J(neqs2, neqs2, 0) , signame1 )

        Bstack =  0.5 * (I(neqs) # J(neqs,1,1)) :* (J(neqs,1,1) # st_matrix("e(B)"))
        Lstack =        (I(neqs) # J(neqs,1,1)) :* (L:^(-0.5)   # J(neqs2,1,1) )
        lincom = ( diag(L:^(0.5)) # I(neqs) , Bstack :* Lstack )
        signame2 = lincom * st_matrix("e(V)") * lincom'
        pRgmBV[2,2] = &( J(neqs2, neqs2, 0) , J(neqs2, neqs2, 0) \ J(neqs2, neqs2, 0) , signame2 )

        // rowvector rgms_calc is sorted
        return(pRgmBV[rgms_calc,.])
    }
end



version 11.2
mata:
mata set matastrict on
    void _dsimih_mergebs() {
    // calculates s.d. of SIRF and SFEVD cols of T_dsimih_bs and column-joins the result to T_dsimih

        string rowvector repchars
        real scalar    numrgms_calc, neqs, hbar, reps, numreprows, numrows, i, first, last, goodreps, num_s
        real colvector rgms_calc
        real matrix    bs, jnk, sdmat, outmat
        pointer(real matrix) scalar    pEst
        pointer(real matrix) colvector pBS
        
        repchars   = tokens(st_local("repchars"))
        rgms_calc  = strtoreal(tokens(st_local("rgms_calc")))
        num_s      = cols(rgms_calc)
        neqs       = cols(st_matrix("e(B)"))
        hbar       = strtoreal(st_local("step"))
        reps       = strtoreal(st_local("reps"))
        numreprows = num_s*neqs^2*(hbar+1)  // rows per replication
        numrows    = numreprows*reps             // total # of rows in T_dsimih_bs
        bs = J(numrows,9,.)
        bs[.,9] = (1::reps) # J(num_s*neqs^2*(hbar+1),1,1)
        pBS = *(findexternal("T_dsimih_bs"))
        for (i=1;i<=reps;i++) {
            first = (numreprows*(i-1))+1
            last  = numreprows*i
            if (repchars[i]==".") {
                bs[|first,1 \ last,8|] = *(pBS[i])
            }
        }

        bs = select(bs, bs[.,5]:<.)
        goodreps = rows(bs) / numreprows
        assert(goodreps==strtoreal(st_local("conv_cnt"))) // TODO: safe check for missings, and whether missings are in line with results saved in e() about bootstrap convergence

        // ops like -collapse (sd)-
        jnk   = rowshape(bs[.,(5,6)], goodreps)
        sdmat = ( bs[1::numreprows,1..4] , colshape(sqrt(diagonal(variance(jnk))'), 2) )  // TODO: variance() calcs the covmat, but I need only column variances ; write loop over columns

        pEst = *(findexternal("T_dsimih"))
        outmat = *pEst
        *pEst = (outmat[.,1..6] , sdmat[.,(5,6)])

    }
end




program define strtokens, rclass

    version 11.0

    syntax namelist(name=passed min=1)            ///
                      , Allowed(namelist min=1)  /// 
                      [ Comblist(string asis)    ///  option combinations allowed ; if not used, all combs are allowed
                        COMBOrder                /// namelist must have tokens whose order corresponds to the one given in -comblist()-
                        Required(string)         ///  NOT YET IMPLEMENTED
                      ]

    local passed : list clean passed  // get rid of multiple spaces
    local dups : list dups passed
    if `"`dups'"'!="" error 198
        
    * local 0 `", `passed'"'
    * syntax , [`allowed']  // returns _rc=198 if tokens in namelist are not allowed

/*
    tokenize `"`allowed'"'
    local expanded ""
    local i 1
    while ("``i''"!="") {
        local il = lower(`"``i''"')
        if `"``il''"'!="" {
            local expanded `expanded' `il'
            local order `order' `i'
        }
        local ++i
    }
*/
    local expanded ""
    foreach curtoken of local allowed {  // copy lower(`allowed') to `allowed_dim' ; use loop to avoid truncation to 244 chars by string function
        local jnk = lower(`"`curtoken'"')
        local allowed_dim `allowed_dim' `jnk'   // "dim": diminishing
    }
    tokenize `"`passed'"'
    local i 1
    while ("``i''"!="") {
        local 0 ", ``i''"
        syntax , [`allowed']
        foreach curallowed of local allowed_dim {
            if "``curallowed''"!="" {
                local expanded `expanded' `curallowed'
                * local order `order' `i'
                local allowed_dim : list allowed_dim - curallowed
                continue, break
            }
        }
        local ++i
    }
/*
    local numexpanded : word count `expanded'
    forvalues i=1/`numexpanded' {
        local pos : word `i' of `order'
        local ordered `ordered' `: word `pos' of `expanded''
    }
    local expanded `ordered'
*/

    local eqset "==="                         // ignores order
    if "`comborder'"!="" local eqset "=="     // does not ignore order

    if `"`comblist'"'!="" {
        local numcomb : word count `comblist'
        forvalues i=1/`numcomb' {
            local curcomb : word `i' of `comblist'
            local match : list expanded `eqset' curcomb
            if `match' continue, break
        }
        if !`match' error 198
    }

    return local passed     `passed'
    return local expanded `expanded'
    

end





program define mac2cond, rclass

version 10.1
syntax varlist(ts)              ///
       , Macro(string asis)          /// 
       [ Comparison(string)     ///
         and                    ///
         or                     ///
         labval]

if `"`comparison'"' != "" {
    if !inlist(`"`comparison'"', "==", "!=", "<", ">", "<=", ">=") {
        disp as error `"Argument to option 'comparison' must be one of "==", "!=", "<", ">", "<=", ">="."'
        exit 9
    }
}
else {
    local comparison "=="
    * local strcomparison "=="
}

if "`and'" != "" & "`or'" != "" {
    disp as error `"Options 'and' and 'or' are mutually exclusive."'
    exit 9
}

local numvars: word count `varlist'
* tokenize `"`macro'"'
local numtokens: word count `macro'

if (`numvars' > 1 & `numtokens' > 1) & (`numvars' != `numtokens') {
    disp as error `"If multiple variables and multiple tokens in option 'macro' are specified, number of variables must match number of tokens."'
    exit 9
}

if `numvars' > 1 & `numtokens' == 1 {
    local logop &
    if "`or'" != "" {
        local logop |
    }

    if "`labval'" != "" {
        disp as error `"When specifying several variables, option 'labval' ist not allowed."'
        exit 9
    }
    // check that either all vars are numeric or all vars are strings
    qui ds `varlist', has(type string)
    local stringlist `r(varlist)'
    if "`stringlist'" == "" {
        local isnumeric true
    }
    else {
        local numstringvars: word count `stringlist'
        if `numstringvars' != `numvars' {
            disp as error `"When specifying several variables and one token in option 'macro', variables must be either all numeric or all string."'
            exit 9
        }
    }

    if "`isnumeric'" == "true" {
        capture confirm number `macro'
        if _rc != 0 & `"`macro'"' != "." {
            disp as error `"Numeric variables specified but token of option 'macro' is string."'
            exit 9
        }
        
        local curlogop ""
        foreach curvar of local varlist {
            local cond `"`cond'`curlogop'`curvar'`comparison'`macro'"'
            local curlogop " `logop' "
        }
    }
    else {
        local curlogop ""
        foreach curvar of local varlist {
            if `"`macro'"' == `""""' | `"`macro'"' == `"`""'"'{
                local cond `"`cond'`curlogop'`curvar'`comparison'`macro'"'
                local curlogop " `logop' "
            }
            else {
                local cond `"`cond'`curlogop'`curvar'`comparison'`"`macro'"'"'
                local curlogop " `logop' "
            }
        }
    }

}

if `numvars' == 1 & `numtokens' >= 1 {
    local logop |
    if "`and'" != "" {
        local logop &
    }

    local myvar `varlist'
    capture confirm numeric variable `myvar'
    if _rc == 0 {
        local isnumeric true
        local vlabelname: value label `myvar'
    }
    if "`isnumeric'" == "true" {
        if "`labval'" != "" & "`vlabelname'" == "" {
            disp as error `"Option 'labval' used but variable `myvar' does not have a value label."'
            exit 9
        }

        local curlogop ""
        forvalues i = 1/`numtokens' {
            local curtoken: word `i' of `macro'
            capture confirm number `curtoken'
            if _rc != 0 & `"`curtoken'"' != "." {
                disp as error `"Numeric variable specified but one or more tokens are strings."'
                exit 9
            }
            if "`labval'" == "" {
                local cond `"`cond'`curlogop'`myvar'`comparison'`curtoken'"'
            }
            else {
                if `"`curtoken'"' == "." {
                    local labstring .
                }
                else {
                    local labstring: label `vlabelname' `curtoken'
                }
                if `"`labstring'"' == "." {      // missing values specified
                    local cond `"`cond'`curlogop'`myvar'`comparison'`curtoken'"'
                }
                else if `"`labstring'"' == "" {  // value label does not have an entry for curtoken: use numeric value
                    local cond `"`cond'`curlogop'`myvar'`comparison'`curtoken'"'
                }
                else {
                    local cond `"`cond'`curlogop'`myvar'`comparison'`"`labstring'"':`vlabelname'"'
                }
            }
            local curlogop " `logop' "
        }
    }
    else {
        local curlogop ""
        forvalues i = 1/`numtokens' {
            local curtoken: word `i' of `macro'
            local cond `"`cond'`curlogop'`myvar'`comparison'`"`curtoken'"'"'
            local curlogop " `logop' "
        }
    }
}

if `numvars' > 1 & `numtokens' == `numvars' {
    local logop &
    if "`or'" != "" {
        local logop |
    }

    if "`labval'" != "" {
        disp as error `"When specifying several variables, option 'labval' ist not allowed."'
        exit 9
    }

    local curtokennum 1
    local curlogop ""
    foreach curvar of local varlist {
        local curtoken: word `curtokennum' of `macro'
        capture confirm numeric variable `curvar'
        if !_rc {  // numeric variable
            capture confirm number `curtoken'
            if _rc & `"`curtoken'"' != "." {
                disp as error `"Numeric variable specified but matching token is string."'
                exit 9
            }
            local cond `"`cond'`curlogop'`curvar'`comparison'`curtoken'"'
        }
        else {     // string variable
            local cond `"`cond'`curlogop'`curvar'`comparison'`"`curtoken'"'"'
        }
        local curlogop " `logop' "
        local `++curtokennum'
    }
}

* local cond: subinstr local cond "   " "" // remove initial "   " or "   " from loop (3 blanks)
return local cond `"`cond'"'

disp as text _newline "Condition returned: " as result `"`cond'"'

end




program define _pub__varsim

    version 11  // not sure when -matrix score- was introduced; -score-, relevant at version 8.0, is now outdated

    syntax varlist ,                ///  endogenous variables in the VAR
                neqs(integer)       ///  # of eqs in VAR
                b(string)           ///  name of coefficient matrix to be used    
                fsamp(varname)      ///
                eqlist(string)      ///
                rgmvar(varlist min=1 max=1) ///
              [ brgm2(string)    ///  second VAR coef vector (for BFanelli method); do not use `b2' as this is used in _byobs
                brgm3(string)    ///  
                brgm4(string)    ///  
                bsp                 ///  parametric bootstrap, otherwise the default residual bootstrap is used
                regimes(numlist)    ///  if empty, regimes are taken from e()
                                    /// -----
                debugkeepvars ]     //  debugging: leave residuals and bootstrapped residuals in data set

    if "`e(cmd)'"!="svarih" exit 301
     
    local vlist "`varlist'"

    if "`regimes'"=="" local regimes `e(regimes)'

    if "`bsp'" == "" {  // non-parametric bootstrap
        forvalues i=1/`neqs' {
            tempname resorig`i'
            qui predict double `resorig`i'' , residuals eq(#`i')
            local resoriglist `resoriglist' `resorig`i''

            tempname bsres`i' 
            qui gen double `bsres`i''=.
            local resvars     "`resvars' `bsres`i'' "
        }

        local sortvars : sortedby
        tempname bsresin bsresin2 rob
        qui gen byte `bsresin'  = 0
        qui gen byte `bsresin2' = .
        qui gen long `rob'      = .

        foreach rgm of local regimes {
        
            qui replace `bsresin' = 0 
            qui replace `bsresin' = 1 if `fsamp' & `rgmvar'==`rgm'
            qui replace `bsresin2' = -1*`bsresin'
            
            sort `bsresin2' `sortvars'
            qui count if `bsresin' == 1

            local robmax = r(N)

            qui replace `rob'=int(1+uniform()*`robmax')  if `bsresin'

            forvalues i = 1/`neqs' {
                qui replace `bsres`i''=`resorig`i''[`rob']  if `bsresin'
                qui count if `bsres`i'' >= . & `bsresin'
                if r(N) > 0 {
                    di as err "error drawing bootstrap sample"
                    exit 498
                }   
            }
        }

        sort `sortvars'
    }
    else {  // parametric boostrap
        forvalues i = 1/`neqs' {
            tempname bsres`i' bsres`i'_tmp
            qui gen double `bsres`i'' = .
            local resvars     "`resvars' `bsres`i'' "          // `resvars' and `resvars_tmp' are related to 'simulated' residuals
            local resvars_tmp "`resvars_tmp' `bsres`i'_tmp' "  // generated by -drawnorm-
                                                               // contents is later copied into individual `resvars', but adjusted for est sample and regime
                                                               // this is then used in the simulation
        }
        
         // get red-form covmats from e()
        if inlist("`e(method)'", "Bacchiocchi", "LLutkepohl") {
            foreach s of local regimes {
                tempname sigma_rgm`s'
                matrix `sigma_rgm`s'' = e(Sigma_rgm`s')
            }
        }
        else if "`e(method)'"=="BFanelli" {
            foreach s of local regimes {
                tempname sigma_rgm`s'
                matrix `sigma_rgm`s'' = e(Sigma_var`s')
            }
        }
        
        local i 1
        tempname cursigma_red
        foreach rgm of local regimes {
            matrix `cursigma_red' = `sigma_rgm`rgm''
            qui drawnorm `resvars_tmp' , cov(`cursigma_red') double  // TODO: this is very inefficient; replace w/ Mata code
            local j 1
            foreach res of local resvars {
                local res_tmp : word `j' of `resvars_tmp'
                qui replace `res' = `res_tmp' if `fsamp'==1 & `rgmvar'==`rgm'
                local ++j
            }
            drop `resvars_tmp'
        }
    }
    
    local j 1
    recast double `vlist'
    if "`e(method)'"=="BFanelli" {
        foreach v of local vlist {
            local eqj : word `j' of `eqlist'
            local a1`j'  "if `rgmvar'[\`i']==1 qui matrix score `v' = `b'        in \`i', eq(#`j') replace"
            local a2`j'  "if `rgmvar'[\`i']==2 qui matrix score `v' = `brgm2' in \`i', eq(#`j') replace"
            local a3`j'  "if `rgmvar'[\`i']==3 qui matrix score `v' = `brgm3' in \`i', eq(#`j') replace"
            local a4`j'  "if `rgmvar'[\`i']==4 qui matrix score `v' = `brgm4' in \`i', eq(#`j') replace"
            local b`j'   "qui replace `v'=`v' + `bsres`j'' in \`i'"
            * local c`j'   "list `v' `bsres`j'' in \`i'"
            local j = `j' + 1
        }
        tempvar n
        qui gen long `n' = _n
        qui su `n' if `fsamp' , meanonly
        forvalues i=`r(min)'/`r(max)' {
            if `fsamp'[`i']==0 continue
            forvalues j=1/`neqs' {
                `a1`j''
                `a2`j''
                `a3`j''
                `a4`j''
                * `c`j''
                `b`j''
                * `c`j''
            }
        }
    }
    else {
        foreach v of local vlist {
            local eqj : word `j' of `eqlist'
            local a`j' "score `v' = `b', eq(#`j') "
            local b`j' "update `v'=`v' + `bsres`j''"
            local j = `j' + 1
        }
        _byobs {
             `a1'
             `b1'
             `a2'
             `b2'
             `a3'
             `b3'
             `a4'
             `b4'
             `a5'
             `b5'
             `a6'
             `b6'
             `a7'
             `b7'
             `a8'
             `b8'
             `a9'
             `b9'
             `a10'
             `b10'
             `a11'
             `b11'
             `a12'
             `b12'
             `a13'
             `b13'
             `a14'
             `b14'
             `a15'
             `b15'
             `a16'
             `b16'
             `a17'
             `b17'
             `a18'
             `b18'
             `a19'
             `b19'
             `a20'
             `b20'
        } if `fsamp' 
    }
    
    // only relevant for cert / debugging purposes
    if "`debugkeepvars'"!="" {

        forvalues i=1/`neqs' {
            capture drop res_eq`i'
            capture drop bsres_eq`i'

            qui gen res_eq`i'   = `resorig`i''
            qui gen bsres_eq`i' = `bsres`i''
        }

        if "`bsp'"=="" {  // residual bootstrap
            capture drop _varsim_rob
            qui gen _varsim_rob = `rob'
        }
        else {
        }
    }

end




version 10
mata:
mata set matastrict on

void ds_pathparts(string scalar origpath) {
// version 1.0.0  01jul2012  dcs
// break full path into parts: path, filename, root (of filename), extension
// store results in r() macros r(root), r(ext), r(filename), r(path)
// 
// rules
// - to get a r(filename), r(ext), r(root), there must be a dot present in the last element of the string supplied
//   multiple dots in filename are allowed; the last one defines the extension
//   if no dot is present in the last element of the string supplied, everything is packed into r(path)
// - to get r(path), there must either be 
//     no dot in the last elem of the path or
//     if a dot is present, there must be a dir separator
// - if a colon is present, it must be preceeded by some string, otherwise the function errors out
// - the first ending directory separator is removed from r(path); so normally r(path) does not end in a dir separator
//   r(path) only contains a separator for the root dir (e.g. "c:\")
//   it also contains separators if multiple separators are passed
//     i.e. passing "mydir//a.lst" will be split into "mydir/" and "a.lst"
// - r(ext) contains a dot as the first character
// - r-values of missing path parts are not returned (e.g. if only the filename is supplied, r(path) is missing)
// - path may contain blanks
// - dots in paths are allowed

    string scalar path,
                  filename,
                  ext,
                  jnk
    real scalar numdots
    
    pragma unset path
    pragma unset filename
    
    pathsplit(origpath, path, filename)
    ext = pathsuffix(origpath)
    if (ext == "") {     // no extension exists => last elem of path is assumed to be part of directory path and not a file name
        path = pathjoin(path, filename)
        filename = ""
    }
    
    st_rclear()
    st_global("r(path)", path)
    st_global("r(filename)", filename)
    st_global("r(ext)", ext)

    // getting root of filename: account for possibility of several dots in filename
    if (filename != "") {
        jnk = subinstr(filename, ".", "")
        numdots = strlen(filename) - strlen(jnk)
        if (numdots == 0) {
            st_global("r(root)", filename)
        }
        if (numdots == 1 & strpos(filename, ".") > 1) {
            st_global("r(root)", substr(filename, 1, strpos(filename, ".") - 1))
        }
        if (numdots > 1) {
            st_global("r(root)", substr(filename, 1, strlen(filename) - strpos(strreverse(filename), ".")))
        }
    }
}
end


